<?php

namespace App\Http\Controllers;

use App\Models\AnnexeAdministrative;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Models\Baraque;
use App\Models\Menage;
use App\Models\BaraqueDocument;
use App\Models\User;
use App\Models\Douar;
use App\Models\Secteur;
use App\Models\SettingsActivites;
use App\Models\Commerce;
use App\Models\Province;
use App\Models\Arrondissement;
use App\Models\Ortophoto;
use Illuminate\Support\Facades\Storage;
use DataTables;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use App\Models\BaraquePA;
use App\Models\BaraqueTitre;
use App\Models\OrtophotoR;


use App\Models\PlanAmenagement;
use App\Models\Mappe;
use Response;
use Carbon\Carbon;
use Exception;
use stdClass;
use Illuminate\Console\Command;
use proj4php\Proj4php;
use proj4php\Proj;
use proj4php\Point;


include(app_path() . '/Http/Controllers/geoPHP.inc');

class BaraquesContorller extends Controller
{
    public function index()
    {
        if (Auth::user()->has_role('banque')) {
            return redirect(route('menages'));
        }
        $douars = Douar::all();
        $secteurs = Secteur::all();
        $annexes_administratives = AnnexeAdministrative::all();
        $arrondissements = Arrondissement::orderBy('nom', 'asc')->get();
        $provinces = Province::orderBy('nom', 'asc')->get();

        if (Auth::user()->has_role('admin,verif,bet,banque,notaire')) {
            return view('baraques.index', [
                'douars' => $douars,
                'secteurs' => $secteurs,
                'annexes_administratives' => $annexes_administratives,
                'arrondissements' => $arrondissements,
                'provinces' => $provinces,
            ]);
        }
    }
    public function show(String $id)
    {
        $baraque = Baraque::with(['menages', 'commerces', 'douar', 'annexe_administrative', 'arrondissement', 'province'])->find($id);
        if ($baraque->active == 0) {
            abort(404);
        }
        $this->check_permission($baraque);
        $baraque->orthophoto = Ortophoto::where('baraque_id', $baraque->id)->get()->first();

        $menages_number = $baraque->menages->count();

        $PHOTO_BARAQUE = BaraqueDocument::where('type_document', "PHOTO_BARAQUE")->where('baraque_id', $baraque->id)->get();

        foreach ($baraque->menages as $menage) {
            $menage->PHOTO_MENAGE_CIN = BaraqueDocument::where('type_document', "PHOTO_MENAGE_CIN")->where('menage_id', $menage->id)->where('baraque_id', $baraque->id)->get();
            $menage->PHOTO_SITUATION_MATRIMONIALE = BaraqueDocument::whereIn('type_document', ["PHOTO_ACT_MARIAGE", "PHOTO_ACT_DIVORCE", "PHOTO_ACT_DECES"])->where('menage_id', $menage->id)->get();

            $menage->PHOTO_CIN_CONJOINT = BaraqueDocument::where('type_document', "PHOTO_CIN_CONJOINT")->where('menage_id', $menage->id)->get();
        }
        foreach ($baraque->commerces as $commerce) {
            $commerce->PHOTO_COMMERCE_CIN = BaraqueDocument::where('type_document', "PHOTO_COMMERCE_CIN")->where('commerce_id', $commerce->id)->where('baraque_id', $baraque->id)->get();
            $commerce->PHOTO_COMMERCE = BaraqueDocument::where('type_document', "PHOTO_COMMERCE")->where('commerce_id', $commerce->id)->where('baraque_id', $baraque->id)->get();
        }


        if (count($baraque->menages) > 0) {
            $baraque->menages[0]->first = 1;
        }

        if (count($baraque->commerces) > 0) {
            $baraque->commerces[0]->first = 1;
        }
        return view('baraques.show', [
            'baraque' => $baraque,
            'PHOTO_BARAQUE' => $PHOTO_BARAQUE,
            'menages_number' => $menages_number,
        ]);
    }

    public function baraques_datatable(Request $request)
    {


        $baraques = Baraque::where('active', 1)->with('menages')->whereNotNull('douar_vf_id')->where('all_menages_verified', 1);
        $filters = json_decode($request->filters);
        foreach ($filters as $filter) {
            $filter->value = trim($filter->value);
            if (isset($filter->params)) {
                $params = explode(',', $filter->params);
                foreach ($params as $param) {
                    if ($param == "remove_spaces") {
                        $filter->value = str_replace(' ', '', $filter->value);
                    }
                }
            }

            if ($filter->column == 'province_id') {
                $douarIds = Douar::where('province_id', '=', $filter->value)->pluck('id');
                $baraques =  $baraques->whereIn('douar_vf_id', $douarIds);
                continue;
            }

            if ($filter->column == 'arrondissement_id') {
                $douarIds = Douar::where('arrondissement_id', '=', $filter->value)->pluck('id');
                $baraques =  $baraques->whereIn('douar_vf_id', $douarIds);
                continue;
            }

            if ($filter->column == 'annexe_administrative_id') {
                $douarIds = Douar::where('annexe_administrative_id', '=', $filter->value)->pluck('id');
                $baraques =  $baraques->whereIn('douar_vf_id', $douarIds);
                continue;
            }

            if ($filter->column == 'douar_id') {

                $baraques =  $baraques->where('douar_vf_id', $filter->value);
                continue;
            }


            if ($filter->column == 'commerce') {
                if ($filter->value == 1) {
                    $baraques = $baraques->whereHas('commerces');
                } elseif ($filter->value == 0) {
                    $baraques = $baraques->whereDoesntHave('commerces');
                }
                continue;
            }
            if ($filter->column == 'cin') {
                $baraques = $baraques->where('menages.chef_menage_cin', '=', $filter->value);
                continue;
            }
            if ($filter->column == 'created_at') {
                $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
                $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));
                $baraques = $baraques->whereBetween($filter->column, [$from, $to]);
            } else {
                if ($filter->table == '') {
                    $baraques = $baraques->where($filter->column, '=', $filter->value);
                } else {
                    $baraques = $baraques->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, '=',  $filter->value);
                    });
                }
            }
        }
        return Datatables::eloquent($baraques)
            ->addColumn('numero', function ($baraque) {
                $vide_html = "";
                if ($baraque->baraque_vide == 1) {
                    $vide_html = "<span style='color:#1c97bf'> (vide)</span>";
                }
                return "<a href=" . asset('') . 'baraques/show/' . $baraque->id . "><span style='color:#c0665b'><i class='fa-regular fa-folder-open mr-2'></i>" .  $baraque->equipe->prefix . "" . $baraque->numero . $vide_html . "</span>";
            })
            ->addColumn('province', function ($baraque) {

                if (is_null($baraque->douar)) {
                    return "-";
                } else {
                    return $baraque->douar->province->nom;
                }
            })
            ->addColumn('arrondissement', function ($baraque) {
                if (is_null($baraque->douar)) {
                    return "-";
                } else {
                    return $baraque->douar->arrondissement->nom;
                }
            })
            ->addColumn('douar', function ($baraque) {
                if (is_null($baraque->douar)) {
                    return "-";
                } else {
                    return $baraque->douar->nom;
                }
            })
            ->addColumn('surface', function ($baraque) {
                return "<span style='color:#c0665b'>" . number_format($baraque->surface, 2, '.', ' ') . "</span><span style='color:#8b8b8b'> m²<span>";
            })
            ->addColumn('menages', function ($baraque) {
                return "<a href='javascript:;'><span style='color:#c0665b'>" . $baraque->menages->count() . "</span><span style='color:#8b8b8b'> ME<span></a>";
            })
            ->addColumn('commerces', function ($baraque) {
                return "<a href='javascript:;'><span style='color:#c0665b'>" . $baraque->commerces->count() . "</span><span style='color:#8b8b8b'> CO<span></a>";
            })
            ->addColumn('etat_avancement', function ($baraque) {
                return "-";
            })
            ->addColumn('relogement', function ($baraque) {
                return "-";
            })
            ->addColumn('attribution', function ($baraque) {
                return "-";
            })
            ->addColumn('demolution', function ($baraque) {

                if ($baraque->baraque_demolie) {
                    return "<span style='color:#ec4401'>Oui</span>";
                } else {
                    return "<span style='color:#6d6d6d'>Non</span>";
                }
            })
            ->addColumn('zone_pa', function ($baraque) {

                if ($baraque->zones_pa->count() == 0) {
                    return "<span style='color:#6d6d6d'>Néant</span>";
                }
                $html = '<select class="form-control select2nc">';
                foreach ($baraque->zones_pa as $zone_pa) {
                    $html .= '<option>' . $zone_pa->pa->zone . '  - ' . $zone_pa->intersection_area . ' m² (' . round(($zone_pa->intersection_area / $baraque->surface) * 100, 2) . ' %)</option>';
                }
                $html .= '</select>';
                return $html;
            })
            ->addColumn('ref_fonciere', function ($baraque) {

                if ($baraque->titres->count() == 0) {
                    return "<span style='color:#6d6d6d'>Néant</span>";
                }
                $html = '<select class="form-control select2nc">';
                foreach ($baraque->titres as $titre) {
                    $html .= '<option>' . $titre->reference->num . "/" .  $titre->reference->indice . '  - ' . $titre->intersection_area . ' m² (' . round(($titre->intersection_area / $baraque->surface) * 100, 2) . ' %)</option>';
                }
                $html .= '</select>';
                return $html;
            })
            ->addColumn('date', function ($baraque) {
                if (is_null($baraque->created_at)) {
                    return "-";
                } else {
                    return Carbon::createFromFormat("Y-m-d H:i:s", $baraque->created_at)->format("d/m/Y");
                }
            })
            ->rawColumns(['numero', 'province', 'arrondissement', 'douar', 'surface', 'menages', 'commerces', 'etat_avancement', 'relogement', 'zone_pa', 'ref_fonciere', 'date', 'demolution', 'attribution'])
            ->only(['numero', 'province', 'arrondissement', 'douar', 'surface', 'menages', 'commerces', 'etat_avancement', 'relogement', 'zone_pa', 'ref_fonciere', 'date', 'demolution', 'attribution'])
            ->tojson();
    }

    public function pa_datatable(Request $request)
    {

        $zones_pa = BaraquePA::where('baraque_id', $request->id);

        return Datatables::eloquent($zones_pa)
            ->addColumn('zone', function ($zone_pa) {
                return "<span style='color:#c0665b'><i class='fa-solid fa-draw-polygon mr-1'></i>" .  $zone_pa->pa->zone . "</span>";
            })
            ->addColumn('surface', function ($zone_pa) {
                return "<span style='color:#c0665b;font-weight:500'>" . $zone_pa->intersection_area . "</span> <span style='color:#6d6d6d'>m²</span>";
            })

            ->rawColumns(['zone', 'surface'])
            ->only(['zone', 'surface'])
            ->tojson();
    }


    public function titres_datatable(Request $request)
    {

        $titres = BaraqueTitre::where('baraque_id', $request->id);

        return Datatables::eloquent($titres)
            ->addColumn('regime', function ($titre) {
                return "<span style='color:#c0665b'><i class='fa-solid fa-file-contract mr-1'></i>" .  $titre->reference->regime . "</span>";
            })
            ->addColumn('reference', function ($titre) {
                return "<span style='color:#c0665b;font-weight:500'>" . $titre->reference->num . "/" . $titre->reference->indice . "</span>";
            })
            ->addColumn('surface', function ($titre) {
                return "<span style='color:#c0665b;font-weight:500'>" . $titre->intersection_area . "</span> <span style='color:#6d6d6d'>m²</span>";
            })

            ->rawColumns(['regime', 'reference', 'surface'])
            ->only(['regime', 'reference', 'surface'])
            ->tojson();
    }

    public function check_permission($baraque)
    {
        if (Auth::user()->has_role('Administrateur')) {
            // if ($dossier->filiale_id != Auth::user()->filiale_id) {
            //     abort(404);
            // }
        }
    }


    function approximateAreaInSquareMeters($geometry)
    {
        $earthRadius = 6378137;

        // Area in square degrees
        $areaDegrees = $geometry->getArea();

        // Get centroid latitude in radians
        $centroid = $geometry->getCentroid();
        $lat = $centroid->getY(); // assuming Y = latitude
        $latRad = deg2rad($lat);

        // Degrees to meters conversion at given latitude
        $meterPerDegreeLat = 111132.92 - 559.82 * cos(2 * $latRad) + 1.175 * cos(4 * $latRad);
        $meterPerDegreeLon = 111412.84 * cos($latRad) - 93.5 * cos(3 * $latRad);

        // Approximate area in square meters
        $areaMeters = $areaDegrees * $meterPerDegreeLat * $meterPerDegreeLon;

        // Return rounded area
        return round($areaMeters, 2);
    }

    public function vsb_migrate(Command $command)
    {
        $command->info('--> Migrate Data from VSB-Maroc');

        self::update_barques_verified($command);
        self::update_barques_douar($command);
        self::update_zone_pa_ref($command);
        self::update_barques_orthophotos($command);
    }


    public function update_barques_verified(Command $command)
    {
        $command->info('--> Update verification status for baraques');

        $baraques = Baraque::where('active', 1)->get();

        $baraques_count = count($baraques);
        $processed = 0;
        $progression = 0;
        $last_progression = 0;
        foreach ($baraques as $baraque) {

            $verified = true;
            foreach ($baraque->menages as  $menage) {
                if ($menage->verrification_etat != "Validé") {
                    $verified = false;
                    break;
                }
            }
            if ($verified) {
                $baraque->all_menages_verified = 1;
            } else {
                $baraque->all_menages_verified = 0;
            }
            $baraque->full_num = $baraque->equipe->prefix . $baraque->numero;
            $baraque->save();

            $processed++;
            $progression = round(($processed / $baraques_count) * 100);
            if ($progression % 2 == 0 && $progression > 0 && $progression != $last_progression) {
                $last_progression = $progression;
                $command->line('    Progression : ' . $progression . " % [ " . $processed . " / " . $baraques_count . " ]");
            }
        }
    }

    public function update_barques_douar(Command $command)
    {
        $command->info('--> Update douars for baraques');

        $geos_installed = \geoPHP::geosInstalled();
        //$command->info($geos_installed);

        $delimitation_douras_path = base_path('files/mapping/delimitations_douars.geojson');

        $delimitation_douras = json_decode(file_get_contents($delimitation_douras_path), true);

        $delimitation_douras_geometries = [];
        $delimitation_douras_geometries_ids = [];

        foreach ($delimitation_douras['features'] as $feature) {
            $geometryJson = json_encode($feature['geometry']);
            $geometry = \geoPHP::load($geometryJson, 'json');

            $delimitation_douras_geometries[] = $geometry;
            $delimitation_douras_geometries_ids[] = $feature["properties"]["id_douar"];
        }
        $baraques = Baraque::where('active', 1)->get();

        $baraques_count = count($baraques);
        $baraques_with_douar = 0;
        $baraques_non_identifiees = 0;


        $processed = 0;
        $progression = 0;
        $last_progression = 0;

        foreach ($baraques as $baraque) {

            $douar_found = 0;
            try {
                $baraque_geometry = \geoPHP::load($baraque->geometry, 'geojson');

                for ($i = 0; $i < count($delimitation_douras_geometries); $i++) {

                    $douar_geometry = $delimitation_douras_geometries[$i];

                    $intersection_result = $douar_geometry->intersection($baraque_geometry);

                    if (!is_null($intersection_result)) {
                        if ($intersection_result->getArea() > 0) {
                            $baraque->douar_vf_id = $delimitation_douras_geometries_ids[$i];
                            $baraque->save();
                            $baraques_with_douar++;
                            $douar_found = 1;
                        }
                    }
                }
            } catch (Exception $ex) {
                //$command->error("geometry error : " . $baraque->id);
            }

            if ($douar_found == 0) {
                $baraques_non_identifiees++;
                $baraque->douar_vf_id = "8e390e87-e25e-4502-862c-3de1ab54c9ec";
                $baraque->save();
            }

            $processed++;
            $progression = round(($processed / $baraques_count) * 100);
            if ($progression % 2 == 0 && $progression > 0 && $progression != $last_progression) {
                $last_progression = $progression;
                $command->line('    Progression : ' . $progression . " % [ " . $processed . " / " . $baraques_count . " ]");
            }
        }
        $command->info("    Baraques : " . $baraques_count . ", with douars : " . $baraques_with_douar . ", non identifiées : " . $baraques_non_identifiees);
    }

    public function update_zone_pa_ref(Command $command)
    {

        $command->info('--> Update Zones PA and Titres for baraques');

        $geos_installed = \geoPHP::geosInstalled();
        // $command->info($geos_installed);
        $baraques = Baraque::where('active', 1)->get();

        $plans_amenagement = PlanAmenagement::all();

        $mappe = Mappe::all();

        $pa_geoometries = [];
        $pa_geoometries_id = [];

        foreach ($plans_amenagement as $pa) {
            $pa_geoometries[] = \geoPHP::load($pa->geometry, 'geojson');
            $pa_geoometries_id[] = $pa->id;
        }

        $mappe_geoometries = [];
        $mappe_geoometries_id = [];

        foreach ($mappe as $titre) {
            $mappe_geoometries[] = \geoPHP::load($titre->geometry, 'geojson');
            $mappe_geoometries_id[] = $titre->id;
        }

        $baraques_count = count($baraques);


        $processed = 0;
        $progression = 0;
        $last_progression = 0;

        foreach ($baraques as $baraque) {

            try {
                BaraquePA::where('baraque_id', $baraque->id)->delete();


                $baraque_geometry = \geoPHP::load($baraque->geometry, 'geojson');
                $baraque_area = self::approximateAreaInSquareMeters($baraque_geometry);
                $baraque_area_degree = $baraque_geometry->getArea();

                $baraque->surface = $baraque_area;
                $baraque->save();
                $index = 0;
                foreach ($pa_geoometries as $pa_geometry) {

                    $intersection_result = $pa_geometry->intersection($baraque_geometry);

                    if (!is_null($intersection_result)) {
                        if ($baraque_area_degree == 0) {
                        } else {
                            $intersection_area =   round($baraque->surface * ($intersection_result->getArea() / $baraque_area_degree), 2);
                            if ($intersection_area > 0) {
                                $baraquePa = new BaraquePA();
                                $baraquePa->id = Str::uuid();
                                $baraquePa->baraque_id = $baraque->id;
                                $baraquePa->pa_id = $pa_geoometries_id[$index];
                                $baraquePa->intersection_area =  $intersection_area;
                                $baraquePa->save();
                            }
                        }
                    }
                    $index++;
                }

                BaraqueTitre::where('baraque_id', $baraque->id)->delete();
                $index = 0;
                foreach ($mappe_geoometries as $titre_geometry) {

                    $intersection_result = $titre_geometry->intersection($baraque_geometry);

                    if (!is_null($intersection_result)) {


                        if ($baraque_area_degree == 0) {
                        } else {
                            $intersection_area =   round($baraque->surface * ($intersection_result->getArea() / $baraque_area_degree), 2);
                            if ($intersection_area > 0) {
                                $baraque_titre = new BaraqueTitre();
                                $baraque_titre->id = Str::uuid();
                                $baraque_titre->baraque_id = $baraque->id;
                                $baraque_titre->titre_id = $mappe_geoometries_id[$index];
                                $baraque_titre->intersection_area =  $intersection_area;
                                $baraque_titre->save();
                            }
                        }
                    }
                    $index++;
                }
            } catch (Exception $ex) {
                //$command->error("geometry error : " . $baraque->id);
            }

            $processed++;
            $progression = round(($processed / $baraques_count) * 100);
            if ($progression % 2 == 0 && $progression > 0 && $progression != $last_progression) {
                $last_progression = $progression;
                $command->line('    Progression : ' . $progression . " % [ " . $processed . " / " . $baraques_count . " ]");
            }
        }
    }

    public function update_barques_orthophotos(Command $command)
    {
        $command->info('--> Update Orthophotos for baraques');

        $geos_installed = \geoPHP::geosInstalled();
        //$command->info($geos_installed);


        $baraques = Baraque::where('active', 1)->get();

        $orthophotos =  OrtophotoR::whereNotNull('bbox')->get();
        $orthophotos_ids = [];
        $orthophotos_geometries = [];

        foreach ($orthophotos as $orthophoto) {

            try {
                $geometry = \geoPHP::load($orthophoto->bbox, 'json');
                $orthophotos_geometries[] = $geometry;
                $orthophotos_ids[] = $orthophoto->id;
            } catch (Exception $ex) {
                $command->error("orthophoto geometry error : " . $orthophoto->id);
            }
        }

        $baraques_count = count($baraques);
        $processed = 0;
        $progression = 0;
        $last_progression = 0;

        foreach ($baraques as $baraque) {


            try {
                $baraque_geometry = \geoPHP::load($baraque->geometry, 'geojson');

                Ortophoto::where('baraque_id', $baraque->id)->delete();
                for ($i = 0; $i < count($orthophotos_geometries); $i++) {

                    $ortho_bbox_geometry = $orthophotos_geometries[$i];

                    $intersection_result = $ortho_bbox_geometry->intersection($baraque_geometry);

                    if (!is_null($intersection_result)) {
                        if ($intersection_result->getArea() > 0) {

                            $ortho = new Ortophoto();
                            $ortho->id = Str::uuid();
                            $ortho->baraque_id = $baraque->id;
                            $ortho->orthophotoR_id =  $orthophotos_ids[$i];
                            $ortho->nom = "";
                            $ortho->save();
                        }
                    }
                }
            } catch (Exception $ex) {
                //$command->error("baraque geometry error : " . $baraque->id);
            }

            $processed++;
            $progression = round(($processed / $baraques_count) * 100);
            if ($progression % 2 == 0 && $progression > 0 && $progression != $last_progression) {
                $last_progression = $progression;
                $command->line('    Progression : ' . $progression . " % [ " . $processed . " / " . $baraques_count . " ]");
            }
        }
    }
}
