<?php

namespace App\Http\Controllers;

use App\Models\AnnexeAdministrative;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Models\Baraque;
use App\Models\Menage;
use App\Models\BaraqueDocument;
use App\Models\User;
use App\Models\Projet;
use App\Models\Douar;
use App\Models\SettingsActivites;
use App\Models\Commerce;
use App\Models\Province;
use App\Models\Immeuble;
use App\Models\Arrondissement;
use App\Models\ProjetDouar;
use Illuminate\Database\QueryException;
use App\Models\ProjetDocument;
use Illuminate\Support\Facades\Storage;
use DataTables;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Response;
use Carbon\Carbon;
use stdClass;
use App\Models\AppartementDouar;

class ProjetsController extends Controller
{

    public function index()
    {
        $arrondissements = Arrondissement::all();
        $provinces = Province::all();

        $promoteurs = User::where('role_id', "e6c37f7b-5825-4192-ae8f-808c2267eef2")->get();
        $projets = Projet::all();
        $immeubles = Immeuble::all();

        return view('projets.index', [
            'provinces' => $provinces,
            'arrondissements' => $arrondissements,
            'promoteurs' => $promoteurs,
            'projets' => $projets,
            'immeubles' => $immeubles,
        ]);
    }

    public function projets_datatable(Request $request)
    {

        $projets = Projet::where('active', 1);
        $filters = json_decode($request->filters);
        foreach ($filters as $filter) {
            $filter->value = trim($filter->value);
            if (isset($filter->params)) {
                $params = explode(',', $filter->params);
                foreach ($params as $param) {
                    if ($param == "remove_spaces") {
                        $filter->value = str_replace(' ', '', $filter->value);
                    }
                }
            }


            if ($filter->column == 'created_at') {
                $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
                $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));
                $projets = $projets->whereBetween($filter->column, [$from, $to]);
            } else {
                if ($filter->table == '') {
                    $projets = $projets->where($filter->column, '=', $filter->value);
                } else {
                    $projets = $projets->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, 'like', '%' . $filter->value . '%');
                    });
                }
            }
        }

        return Datatables::eloquent($projets)
            ->addColumn('projet', function ($projet) {
                return "<a href=" . asset('') . 'projets/show/' . $projet->id . "><span style='color:#c0665b'><i class='fa-solid fa-building-circle-arrow-right mr-2'></i>" . $projet->nom . "</span>";
            })

            ->addColumn('province', function ($projet) {
                return $projet->province->nom;
            })



            ->addColumn('promoteur', function ($projet) {
                return $projet->promoteur->societe;
            })

            ->addColumn('agence', function ($projet) {
                return "-";
            })
            ->addColumn('banque', function ($projet) {
                return "-";
            })
            ->addColumn('rib', function ($projet) {
                return "-";
            })
            ->addColumn('actions', function ($projet) {
                return "-";
            })

            ->addColumn('nbr_immeubles', function ($projet) {
                return $projet->immeubles->count();
            })

            ->addColumn('nbr_appartements', function ($projet) {
                return $projet->appartements->count();
            })

            ->addColumn('appartements_affectes', function ($projet) {
                return $projet->appartements->where('is_affected', 1)->count();
            })

            ->addColumn('taux_affectation', function ($projet) {
                $taux = round(($projet->appartements->where('is_affected', 1)->count() / $projet->appartements->count()) * 100, 2);
                return '<div class="progress mb-1">
                                    <div   class="progress-bar bg-info" role="progressbar" style="width: ' . $taux . '%;" aria-valuenow="' . $taux . '" aria-valuemin="0" aria-valuemax="100"> <span style="float: right;">' . $taux . ' %</span></div>

                                </div>';
            })

            ->rawColumns(['agance', 'banque', 'projet', 'rib', 'actions', 'province',  'promoteur', 'nbr_immeubles', 'nbr_appartements', 'appartements_affectes', 'taux_affectation'])
            ->only(['agance', 'banque', 'projet', 'rib', 'actions', 'projet', 'province', 'promoteur', 'nbr_immeubles', 'nbr_appartements', 'appartements_affectes', 'taux_affectation'])
            ->tojson();
    }


    public function show(String $id)
    {
        $projet = Projet::where('active', 1)->where("id", $id)->get()->first();

        $annexes_administratives = AnnexeAdministrative::all();
        $arrondissements = Arrondissement::orderBy('nom', 'asc')->get();
        $provinces = Province::orderBy('nom', 'asc')->get();
        $douars = Douar::orderBy('nom', 'asc')->get();


        if ($projet->active == 0) {
            abort(404);
        }
        $this->check_permission($projet);
        $taux_affectation = 0;
        if ($projet->appartements->count() > 0) {
            $taux_affectation = round(($projet->appartements->where('is_affected', 1)->count() / $projet->appartements->count()) * 100, 2);
        }


        $CONVENTION_doc = [
            'type_document' => 'CONVENTION',
            'name' => "Convention",
            'document' => $projet->specific_document('CONVENTION'),
            'file_type' => 'pdf',
            'w_roles' => 'promoteur,admin',
            'r_roles' => 'all',
            'active' => $projet->document_can_be_imported('CONVENTION'),
            'validation' => 1,
            'date_doc' => 0
        ];


        $AUTORISATION_doc = [
            'type_document' => 'AUTORISATION',
            'name' => "Autorisation",
            'document' => $projet->specific_document('AUTORISATION'),
            'file_type' => 'pdf',
            'w_roles' => 'promoteur,admin',
            'r_roles' => 'all',
            'active' =>  $projet->document_can_be_imported('AUTORISATION'),
            'validation' => 1,
            'date_doc' => 0
        ];

        $OUVERTURE_CHANTIER_doc = [
            'type_document' => 'OUVERTURE_CHANTIER',
            'name' => "Ouverture de chantier",
            'document' => $projet->specific_document('OUVERTURE_CHANTIER'),
            'file_type' => 'pdf',
            'w_roles' => 'promoteur,admin',
            'r_roles' => 'all',
            'active' => $projet->document_can_be_imported('OUVERTURE_CHANTIER'),
            'validation' => 1,
            'date_doc' => 0,
            'date_previsionnelle' => $projet->ouverture_chantier_date_previsionnelle,

        ];


        $OUVERTURE_CHANTIER_photos = [
            'type_document' => 'OUVERTURE_CHANTIER_PHOTO',
            'name' => "Photo de suivi",
            'title' => "",
            'photos' => $projet->specific_photos('OUVERTURE_CHANTIER_PHOTO'),
            'file_type' => 'png jpeg jpg bmp'
        ];






        foreach ($projet->tranches as $tranche) {


            $TRANCHE_docs = [];

            $TRANCHE_docs[] = [
                'type_document' => 'TR_RECEPTION_PROVISOIRE_' . $tranche->numero,
                'name' => "Réception provisoire",
                'document' => $projet->specific_document('TR_RECEPTION_PROVISOIRE_' . $tranche->numero),
                'file_type' => 'pdf',
                'w_roles' => 'promoteur,admin',
                'r_roles' => 'all',
                'active' => $projet->document_can_be_imported('TR_RECEPTION_PROVISOIRE_' . $tranche->numero),
                'validation' => 1,
                'date_doc' => 0,
                'tranche' => $tranche->numero,
                'tranche_id' => $tranche->id,
                'date_previsionnelle' => $tranche->reception_provisoire_date_previsionnelle,
            ];


            $TRANCHE_docs[] = [
                'type_document' => 'TR_ECLATEMENT_TITRES_' . $tranche->numero,
                'name' => "Eclatement des titres fonciers",
                'document' => $projet->specific_document('TR_ECLATEMENT_TITRES_' . $tranche->numero),
                'file_type' => 'pdf',
                'w_roles' => 'promoteur,admin',
                'r_roles' => 'all',
                'active' => $projet->document_can_be_imported('TR_ECLATEMENT_TITRES_' . $tranche->numero),
                'validation' => 1,
                'date_doc' => 0,
                'tranche' => $tranche->numero,
                'tranche_id' => $tranche->id,
                'date_previsionnelle' => $tranche->eclatement_titres_date_previsionnelle,
            ];


            $TRANCHE_docs[] = [
                'type_document' => 'TR_PERMIS_HABITER_' . $tranche->numero,
                'name' => "Permis d'habiter",
                'document' => $projet->specific_document('TR_PERMIS_HABITER_' . $tranche->numero),
                'file_type' => 'pdf',
                'w_roles' => 'promoteur,admin',
                'r_roles' => 'all',
                'active' =>  $projet->document_can_be_imported('TR_PERMIS_HABITER_' . $tranche->numero),
                'validation' => 1,
                'date_doc' => 0,
                'tranche' => $tranche->numero,
                'tranche_id' => $tranche->id,
                'date_previsionnelle' => $tranche->permis_habite_date_previsionnelle,
            ];



            $tranche->TRANCHE_docs =  $TRANCHE_docs;
            $tranche->first = false;
        }
        $projet->tranches[0]->first = true;





        return view('projets.show', [
            'projet' => $projet,
            'taux_affectation' => $taux_affectation,
            'provinces' => $provinces,
            'annexes_administratives' => $annexes_administratives,
            'arrondissements' => $arrondissements,
            'douars' => $douars,
            'CONVENTION_doc' => $CONVENTION_doc,
            'AUTORISATION_doc' => $AUTORISATION_doc,
            'OUVERTURE_CHANTIER_doc' => $OUVERTURE_CHANTIER_doc,
            'OUVERTURE_CHANTIER_photos' => $OUVERTURE_CHANTIER_photos,


        ]);
    }

    public function store(Request $request)
    {
        $projet = Projet::where('id', $request->id)->get()->first();

        $dates_previsionnelles = json_decode($request->dates_previsionnelles);

        $projet->phase_etude = $request->phase_etude;
        if (!empty($request->date_operation_phase_etude)) {
            $projet->date_operation_phase_etude =  Carbon::createFromFormat('d/m/Y', $request->date_operation_phase_etude)->format('Y-m-d H:i:s');
        }
        $projet->permis_habiter = $request->permis_habiter;
        if (!empty($request->date_operation_permis_habiter)) {
            $projet->date_operation_permis_habiter =  Carbon::createFromFormat('d/m/Y', $request->date_operation_permis_habiter)->format('Y-m-d H:i:s');
        }

        foreach ($dates_previsionnelles as $date_previsionnelle) {
            if ($date_previsionnelle->tranche_id == "all") {
                if ($date_previsionnelle->type_document == "OUVERTURE_CHANTIER") {
                    $projet->ouverture_chantier_date_previsionnelle = $date_previsionnelle->value;
                }
            } else {

                foreach ($projet->tranches as $tranche) {

                    if ($tranche->id == $date_previsionnelle->tranche_id) {
                        if (str_starts_with($date_previsionnelle->type_document,"TR_RECEPTION_PROVISOIRE_")) {
                            $tranche->reception_provisoire_date_previsionnelle = $date_previsionnelle->value;
                        }
                        if (str_starts_with($date_previsionnelle->type_document,"TR_ECLATEMENT_TITRES_")) {
                            $tranche->eclatement_titres_date_previsionnelle = $date_previsionnelle->value;
                        }
                       if (str_starts_with($date_previsionnelle->type_document,"TR_PERMIS_HABITER_")) {
                            $tranche->permis_habite_date_previsionnelle = $date_previsionnelle->value;
                        }
                        $tranche->save();
                    }
                }
            }
        }

        //$projet->phase_etude = $request->phase_etude;
        $projet->save();

        return Response::json(['success' => 1], 201);
    }

    public function get_projets($id = null)
    {
        if ($id) {
            $projet = Projet::with(['immeubles.appartements' => function ($query) {
                $query->where('is_affected', 0);
            }])->find($id);

            if (!$projet) {
                return Response::json(['success' => 0, 'message' => 'Projet non trouvé'], 404);
            }

            return Response::json(['success' => 1, 'projets' => $projet]);
        }

        $projets = Projet::with(['immeubles.appartements' => function ($query) {
            $query->where('is_affected', 0);
        }])->get();
        return Response::json(['success' => 1, 'projets' => $projets]);
    }



    public function file(string $file_id)
    {
        $projet_doc = ProjetDocument::where('id', $file_id)->get()->first();
        if (is_null($projet_doc)) {
            abort(404);
        } else {
            $headers = array(
                'Content-Type: application/pdf',
            );
            return Response::download($projet_doc->documentPath(), $projet_doc->filename, $headers, "inline");
        }
    }
    public function file_data(string $file_id)
    {
        $projet_document = ProjetDocument::where('id', $file_id)->get()->first();
        return $projet_document;
    }
    public function upload(Request $request)
    {
        $now = Carbon::now();
        $now->hour(23);
        $now->minute(59);
        $now->second(59);
        $projet = Projet::find($request->projet_id);
        $projet_document = new ProjetDocument();
        if ($request->id != null) {
            $projet_document = ProjetDocument::where('id', $request->id)->get()->first();
        } else {
            $projet_document->id = Str::uuid();
        }
        $file = $request->file('file');
        if ($file == null) {
        } else {
            $fileName = $file->getClientOriginalName();
            $file_extension = Str::upper($file->getClientOriginalExtension());
            $file_size = $file->getSize();
        }
        $projet_document->projet_id = $projet->id;
        $projet_document->filename = $fileName;
        $projet_document->nom =  $fileName;
        $projet_document->user_id =  Auth::user()->id;
        $check_for_date_doc = 1;

        // if ($request->date_previsionnelle != null) {
        //     $date_previsionnelle = Carbon::createFromFormat('d/m/Y', $request->date_document);
        //     $projet_document->date_previsionnelle = $date_previsionnelle->format('Y-m-d');
        // }


        if (str_ends_with($request->type_document, '_PHOTO')) {
            $check_for_date_doc = 0;
        }
        if ($check_for_date_doc == 1) {
            if ($request->date_document == null) {
                return Response::json(['success' => 0, 'error' => 'EMPTY_DATE_DOCUMENT']);
            } else {
                $date_doc = Carbon::createFromFormat('d/m/Y', $request->date_document);
                $date_doc->hour(00);
                $date_doc->minute(00);
                $date_doc->second(00);
                if ($date_doc->gt($now)) {
                    return Response::json(['success' => 0, 'error' => 'DATE_IN_FUTURE']);
                }
                $projet_document->date_document = $date_doc->format('Y-m-d');
            }
        } else {
            $projet_document->date_document = $now->format('Y-m-d');
        }
        $projet_document->file_type = $file_extension;
        $projet_document->file_size = $file_size;

        $projet_document->decision = 'Validé';

        $projet_document->type_document = $request->type_document;
        if ($request->type_document == "RECEPISSE") {
            $projet_document->guichet_id = $request->guichet;
        }
        if ($request->type_document == "PLAN_ARCHITECTE_PDF") {
            $projet_document->plan_type_id = $request->plan_type;
        }
        if ($request->type_document == "LIMITE_PLLE") {
            try {
                $geojson = file_get_contents($file->getRealPath());
                $geometry = \geoPHP::load($geojson, 'geojson');
                $centroid = $geometry->getCentroid();
                $wgs84 = false;
                if (abs($centroid->getY()) <= 180 && abs($centroid->getY()) >= 0) {
                    if (abs($centroid->getX()) <= 180 && abs($centroid->getX()) >= 0) {
                        $wgs84 = true;
                    }
                }
                if (!$wgs84) {
                    return Response::json(['success' => 0, 'error' => 'NOT_WGS84']);
                }
            } catch (Exception $ex) {
                return Response::json(['success' => 0, 'error' => 'WRONG_GEOJSON_FILE']);
            }
        }
        $projet_document->save();
        if ($request->date_document != null) {
            $projet_document->formated_date_document = $projet_document->formated_date_document();
        }
        $projet_document->formated_created_at = $projet_document->formated_created_at();
        $file->move("files/projets_documents/" . $projet_document->id . "/", $fileName);

        $create_thumbnail = 1;
        if ($create_thumbnail == 1) {
            $resolution = 150;
            $width = 500;

            $pdfPathEscaped = escapeshellarg("{$projet_document->documentPath()}[0]");
            $outputJpgEscaped = escapeshellarg($projet_document->documentThumbnailPath());

            $cmd = "convert -density $resolution $pdfPathEscaped -background white -alpha remove -alpha off -quality 90 -resize {$width} $outputJpgEscaped";

            exec($cmd, $output, $return_var);
        }
        $projet_document->url = $projet_document->documentUrl();
        return Response::json(['success' => 1, 'document' => $projet_document]);
    }


    public function check_permission($projet)
    {
        if (Auth::user()->has_role('Administrateur')) {

            // if ($dossier->filiale_id != Auth::user()->filiale_id) {

            //     abort(404);

            // }

        }
    }

    public function douars_affectes_datatable(Request $request)
    {
        $projet_douars = Projet::where('id', $request->projet_id)->get()->first()->projet_douars();

        $filters = json_decode($request->filters);
        foreach ($filters as $filter) {
            $filter->value = trim($filter->value);
            if (isset($filter->params)) {
                $params = explode(',', $filter->params);
                foreach ($params as $param) {
                    if ($param == "remove_spaces") {
                        $filter->value = str_replace(' ', '', $filter->value);
                    }
                }
            }



            if ($filter->column == 'created_at') {

                $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
                $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));

                $projets = $projets->whereBetween($filter->column, [$from, $to]);
            } else {
                if ($filter->table == '') {
                    $projets = $projets->where($filter->column, '=', $filter->value);
                } else {
                    $projets = $projets->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, 'like', '%' . $filter->value . '%');
                    });
                }
            }
        }

        return Datatables::of($projet_douars)
            ->addColumn('douar', function ($projet_douar) {
                return '<span style="color:#c0665b;font-weight:500"><i class="fa-solid fa-location-dot mr-1"></i>' . $projet_douar->douar->nom . "</span>";
            })
            ->addColumn('annexe_administrative', function ($projet_douar) {
                return  $projet_douar->douar->annexe_administrative->nom;
            })
            ->addColumn('arrondissement', function ($projet_douar) {
                return  $projet_douar->douar->arrondissement->nom;
            })
            ->addColumn('province', function ($projet_douar) {
                return  $projet_douar->douar->arrondissement->province->nom;
            })
            ->addColumn('nbr_appartements', function ($projet_douar) {
                return  $projet_douar->nbr_appartements;
            })
            ->rawColumns(['nbr_appartements', 'douar', 'annexe_administrative', 'arrondissement', 'province'])
            ->only(['nbr_appartements', 'douar', 'annexe_administrative', 'arrondissement', 'province'])
            ->tojson();
    }

    public function assigner_douar(Request $request)
    {
        $request->appartments_ids = explode(',', $request->appartments_ids);

        foreach ($request->appartments_ids as $appartment_id) {

            $appartment_douar = new AppartementDouar();

            $appartment_douar->id = Str::uuid();
            $appartment_douar->douar_id = $appartment_id;
            $appartment_douar->douar_id = $request->douar_id;

            $appartment_douar->projet_id = $request->projet_id;
            $appartment_douar->appartment_id = $appartment_id;

            $appartment_douar->hash = $request->douar_id . "-" . $appartment_id;


            try {
                $appartment_douar->save();
            } catch (QueryException $e) {
                // Check if the error is due to unique constraint violation
                if ($e->getCode() == 23000) {

                    continue;
                }

                // Other errors, rethrow
                throw $e;
            }
        }
    }


    public function formated_ouvetrure_date_previsionnelle()
    {
        if (is_null($this->ouvetrure_date_previsionnelle))
            return "-";
        return Carbon::createFromFormat('Y-m-d', $this->ouvetrure_date_previsionnelle)->format('d/m/Y');
    }
}
