<?php

namespace App\Http\Controllers;


use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;

use App\Models\Ratissage;
use App\Models\RatissageDocument;
use App\Models\Resident;
use App\Models\Risque;
use App\Models\MesureUrgence;
use App\Models\ResidentDocument;
use App\Models\Dossier;
use App\Models\User;


use App\Models\MenageDocument;
use App\Models\ChefMenage;
use App\Models\Menage;


use App\Models\Province;
use App\Models\Arrondissement;
use App\Models\AnnexeAdministrative;
use App\Models\DossierLog;
use App\Models\Quartier;
use Illuminate\Support\Facades\Storage;
use DataTables;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Response;
use Carbon\Carbon;
use stdClass;

use Spatie\PdfToImage\Pdf;



class RatissagesController extends Controller
{
    public function index()
    {
        $provinces = Province::orderBy('nom', 'asc')->get();
        $arrondissements = Arrondissement::orderBy('nom', 'asc')->get();
        $annexes_administratives = AnnexeAdministrative::orderBy('nom', 'asc')->get();
        $quartiers = Quartier::orderBy('nom', 'asc')->get();
        // if (Auth::user()->has_role('admin')) {
        return view('ratissages.index', [
            'provinces' => $provinces,
            'arrondissements' => $arrondissements,
            'annexes_administratives' => $annexes_administratives,
            'quartiers' => $quartiers,
        ]);
        // }
    }

    public function ratissages_datatable(Request $request)
    {
        $ratissages = Ratissage::with('menages')->where('active', 1);

        $searchTerm = $request->searchTerm;

        $referer = "";
        if (isset($request->referer)) {
            $referer = $request->referer;
        }
        $highlighted_document = null;
        $verificateur = 'all';
        if (isset($request->verificateur)) {
            $verificateur = $request->verificateur;
        }
        $filters = json_decode($request->filters);
        foreach ($filters as $filter) {
            $filter->value = trim($filter->value);
            if (isset($filter->params)) {
                $params = explode(',', $filter->params);
                foreach ($params as $param) {
                    if ($param == "remove_spaces") {
                        $filter->value = str_replace(' ', '', $filter->value);
                    }
                }
            }

            if ($filter->column == 'province_id') {
                $ratissages =  $ratissages->where('province_id', $filter->value);
                continue;
            }


            if ($filter->column == 'arrondissement_id') {
                $ratissages =  $ratissages->where('arrondissement_id', $filter->value);
                continue;
            }

            if ($filter->column == 'annexe_administrative_id') {
                $ratissages =  $ratissages->where('annexe_administrative_id', $filter->value);
                continue;
            }


            if ($filter->column == 'quartier_id') {
                $ratissages =  $ratissages->where('quartier_id', $filter->value);
                continue;
            }



            if ($filter->column == 'numero') {
                $ratissages = $ratissages->where('numero', 'LIKE', '%' . $filter->value . '%');
                continue;
            }


            if ($filter->column == 'adresse') {
                $ratissages = $ratissages->where('adresse', 'LIKE', '%' . $filter->value . '%');
                continue;
            }

            if ($filter->column == 'source') {
                $ratissages = $ratissages->where('source', 'LIKE', '%' . $filter->value . '%');
                continue;
            }

            if ($filter->column == 'occupation') {
                $ratissages = $ratissages->where('occupation', 'LIKE',  $filter->value);
                continue;
            }


            if ($filter->column == 'usage') {
                $ratissages = $ratissages->where('usage_batisse', 'LIKE',  $filter->value);
                continue;
            }

            if ($filter->column == 'typologie') {
                $ratissages = $ratissages->where('typologie', 'LIKE',  $filter->value);
                continue;
            }
        }

        return Datatables::eloquent($ratissages)

            ->addColumn('numero', function ($ratissage) {
                return "<a href=" . asset('') . 'ratissages/show/' . $ratissage->id . "><span style='color:#c0665b'><i class='fa-regular fa-folder-open mr-2'></i>" .  $ratissage->numero . "</span>";
            })

            ->addColumn('adresse', function ($ratissage) {
                return '<span style="">' . $ratissage->adresse . '</span>';
            })
            ->addColumn('menages', function ($ratissage) {
                return "<a href='javascript:;'><span style='color:#c0665b'>" . $ratissage->menages->count() . "</span><span style='color:#8b8b8b'> ME<span></a>";
            })

            ->addColumn('arrondissement', function ($ratissage) {
                if (is_null($ratissage->arrondissement)) {
                    return "-";
                } else {
                    return $ratissage->arrondissement->nom;
                }
            })

            ->addColumn('annexe_administrative', function ($ratissage) {
                if (is_null($ratissage->annexe_administrative)) {
                    return "-";
                } else {
                    return $ratissage->annexe_administrative->nom;
                }
            })
            ->addColumn('quartier', function ($ratissage) {
                if (is_null($ratissage->quartier)) {
                    return "-";
                } else {
                    return $ratissage->quartier->nom;
                }
            })

            ->addColumn('date', function ($ratissage) {
                return '<span style="">' . $ratissage->date . '</span>';
            })

            ->rawColumns(['numero', 'menages', 'adresse', 'arrondissement', 'annexe_administrative', 'quartier', 'date'])
            ->only(['numero', 'menages', 'adresse', 'arrondissement', 'annexe_administrative', 'quartier', 'date'])
            ->tojson();
    }


    public function show(String $id)
    {
        $ratissage = Ratissage::with([
            'menages.chefs_menage', // Add chefs_menage here
            'commerces',
            'province',
            'annexe_administrative',
            'quartier'
        ])->find($id);

        if (!$ratissage || $ratissage->active == 0) {
            abort(404);
        }

        $menages_number = $ratissage->menages->count();

        $ARRETE_DEMOLITION_document = [
            'type_document' => 'ARRETE_DEMOLITION',
            'name' => "Expertise",
            'document' => $ratissage->specific_document('ARRETE_DEMOLITION'),
            'file_type' => 'pdf',
            'w_roles' => 'all',
            'r_roles' => 'all',
            'active' => $ratissage->document_can_be_imported('ARRETE_DEMOLITION'),
            'validation' => 0,
            'date_doc' => 0,
            'show' => 1,
        ];

        // Loop through all menages
        foreach ($ratissage->menages as $menage) {
            foreach ($menage->chefs_menage as $chef_menage) {
                $chef_menage_files_list = [];

                // Basic documents for all chefs de menage
                $chef_menage_files_list[] = [
                    'type_document' => 'CM_MENAGE_CIN',
                    'name' => "CIN Chef de ménage",
                    'document' => $chef_menage->specific_document('CM_MENAGE_CIN'),
                    'file_type' => 'pdf',
                    'w_roles' => '',
                    'r_roles' => 'all',
                    'active' => $chef_menage->document_can_be_imported('CM_MENAGE_CIN'),
                    'validation' => 1,
                    'date_doc' => 0,
                    'show' => 1,
                    'break' => 0
                ];

                $chef_menage_files_list[] = [
                    'type_document' => 'ATTESTATION_DEMOLITION',
                    'name' => "Attestation de démolition",
                    'document' => $chef_menage->specific_document('ATTESTATION_DEMOLITION'),
                    'file_type' => 'pdf',
                    'w_roles' => 'notaire',
                    'r_roles' => 'all',
                    'active' => $chef_menage->document_can_be_imported('ATTESTATION_DEMOLITION'),
                    'validation' => 1,
                    'date_doc' => 0,
                    'show' => 1,
                    'break' => 0
                ];

                // Documents for divorced status
                if ($chef_menage->situation == "Divorcé") {
                    $chef_menage_files_list[] = [
                        'type_document' => 'CM_ACT_DIVORCE',
                        'name' => "Acte de divorce",
                        'document' => $chef_menage->specific_document('CM_ACT_DIVORCE'),
                        'file_type' => 'pdf',
                        'w_roles' => '',
                        'r_roles' => 'all',
                        'active' => 1,
                        'validation' => $chef_menage->document_can_be_imported('CM_ACT_DIVORCE'),
                        'date_doc' => 0,
                        'show' => 1,
                        'break' => 0
                    ];
                }

                // Documents for widowed status
                if ($chef_menage->situation == "Veuf") {
                    $chef_menage_files_list[] = [
                        'type_document' => 'CM_ACT_DECES',
                        'name' => "Acte de décès",
                        'document' => $chef_menage->specific_document('CM_ACT_DECES'),
                        'file_type' => 'pdf',
                        'w_roles' => '',
                        'r_roles' => 'all',
                        'active' => 1,
                        'validation' => $chef_menage->document_can_be_imported('CM_ACT_DECES'),
                        'date_doc' => 0,
                        'show' => 1,
                        'break' => 0
                    ];
                }

                // Store the chef_menage files list
                $chef_menage->chef_menage_files_list = $chef_menage_files_list;

                // Handle conjoint documents for married status
                $conjoint_chef_menage_files_list = [];

                if ($chef_menage->situation == "Marié") {
                    foreach ($chef_menage->conjoints() as $conjoint) {
                        // CIN Conjoint
                        $conjoint_chef_menage_files_list[] = [
                            'type_document' => 'CM_CIN_CONJOINT_' . $conjoint->index,
                            'name' => "CIN conjoint",
                            'document' => $chef_menage->specific_document('CM_CIN_CONJOINT_' . $conjoint->index),
                            'file_type' => 'pdf',
                            'w_roles' => '',
                            'r_roles' => 'all',
                            'active' => 1,
                            'validation' => $chef_menage->document_can_be_imported('CM_CIN_CONJOINT_' . $conjoint->index),
                            'date_doc' => 0,
                            'show' => 1,
                            'break' => 0,
                            'CONJOINT_index' => $conjoint->index,
                            'CONJOINT_info' => $conjoint // Store conjoint info for display
                        ];

                        // Acte de Mariage
                        $conjoint_chef_menage_files_list[] = [
                            'type_document' => 'CM_ACT_MARIAGE_' . $conjoint->index,
                            'name' => "Acte de mariage",
                            'document' => $chef_menage->specific_document('CM_ACT_MARIAGE_' . $conjoint->index),
                            'file_type' => 'pdf',
                            'w_roles' => '',
                            'r_roles' => 'all',
                            'active' => 1,
                            'validation' => $chef_menage->document_can_be_imported('CM_ACT_MARIAGE_' . $conjoint->index),
                            'date_doc' => 0,
                            'show' => 1,
                            'break' => 0,
                            'CONJOINT_index' => $conjoint->index,
                            'CONJOINT_info' => $conjoint
                        ];

                        // Attestation de démolition Conjoint (if applicable)
                        $conjoint_chef_menage_files_list[] = [
                            'type_document' => 'ATTESTATION_DEMOLITION_CONJOINT_' . $conjoint->index,
                            'name' => "Attestation de démolition conjoint",
                            'document' => $chef_menage->specific_document('ATTESTATION_DEMOLITION_CONJOINT_' . $conjoint->index),
                            'file_type' => 'pdf',
                            'w_roles' => 'notaire',
                            'r_roles' => 'all',
                            'active' => $chef_menage->document_can_be_imported('ATTESTATION_DEMOLITION_CONJOINT_' . $conjoint->index),
                            'validation' => 1,
                            'date_doc' => 0,
                            'show' => 1,
                            'break' => 1, // Add break line after last document of this conjoint
                            'CONJOINT_index' => $conjoint->index,
                            'CONJOINT_info' => $conjoint
                        ];
                    }
                }

                // Store the conjoint files list
                $chef_menage->conjoint_chef_menage_files_list = $conjoint_chef_menage_files_list;
            }
        }

        return view('ratissages.show', [
            'ratissage' => $ratissage,
            'ARRETE_DEMOLITION_document' => $ARRETE_DEMOLITION_document,
        ]);
    }



    public function store(Request $request)
    {
        $ratissage = Ratissage::where('id', $request->id)->get()->first();

        $documents = json_decode($request->documents);
        $documents_to_delete = json_decode($request->documents_to_delete);

        // Save uploaded documents
        foreach ($documents as $document) {
            $ratissage_document = RatissageDocument::where('id', $document->id)->get()->first();
            if (!is_null($ratissage_document)) {
                $ratissage_document->ratissage_id = $ratissage->id;
                $ratissage_document->save();
            }
        }

        // Delete marked documents
        foreach ($documents_to_delete as $document) {
            $ratissage_document = RatissageDocument::where('id', $document)->get()->first();
            if ($ratissage_document) {
                // Delete physical files
                $filePath = $ratissage_document->documentPath();
                if (file_exists($filePath)) {
                    unlink($filePath);
                }
                $thumbnailPath = $ratissage_document->documentThumbnailPath();
                if (file_exists($thumbnailPath)) {
                    unlink($thumbnailPath);
                }
                // Delete directory if empty
                $dir = dirname($filePath);
                if (is_dir($dir) && count(scandir($dir)) == 2) {
                    rmdir($dir);
                }
                $ratissage_document->delete();
            }
        }

        $ratissage->save();

        // ============================================
        // CHECK FOR VALIDATED DOCUMENTS AND UPDATE is_dossier
        // ============================================

        // Check if ARRETE_DEMOLITION exists and is validated
        $arrete_demolition = RatissageDocument::where('ratissage_id', $ratissage->id)
            ->where('type_document', 'ARRETE_DEMOLITION')
            ->where('decision', 'Validé')
            ->first();

        // Loop through all menages to check ATTESTATION_DEMOLITION
        foreach ($ratissage->menages as $menage) {
            $should_update_dossier = false;

            // Only proceed if ARRETE_DEMOLITION is validated
            if ($arrete_demolition) {
                // Check all chefs_menage for this menage
                foreach ($menage->chefs_menage as $chef_menage) {
                    // Check if ATTESTATION_DEMOLITION exists and is validated
                    $attestation_demolition = MenageDocument::where('chef_menage_id', $chef_menage->id)
                        ->where('type_document', 'ATTESTATION_DEMOLITION')
                        ->where('decision', 'Validé')
                        ->first();

                    if ($attestation_demolition) {
                        $should_update_dossier = true;
                        break; // At least one chef has validated attestation
                    }
                }
            }

            // Update is_dossier status
            if ($should_update_dossier) {
                if ($menage->is_dossier != 1) {
                    $menage->is_dossier = 1;
                    $menage->save();
                    \Log::info("Menage {$menage->id} is_dossier updated to 1");
                }
            } else {
                // Reset to 0 if conditions are not met anymore
                if ($menage->is_dossier == 1) {
                    $menage->is_dossier = 0;
                    $menage->save();
                    \Log::info("Menage {$menage->id} is_dossier reset to 0");
                }
            }
        }

        // ============================================
        // CREATE DOSSIER IF EXPERTISE EXISTS (commented code preserved)
        // ============================================

        // $att_expertise_doc = RatissageDocument::where('ratissage_id', $ratissage->id)
        //     ->where('type_document', 'ATT_EXPERTISE')
        //     ->first();

        // $dossier_exists = Dossier::where('ratissage_id', $ratissage->id)->exists();

        // if ($att_expertise_doc && !$dossier_exists) {
        //     $dossier = new Dossier();
        //     $dossier->id = Str::uuid();
        //     // You can use a pattern like 'DOS-' plus current year + random or increment
        //     $dossier->numero = $dossier->generateNumero();
        //     $dossier->ratissage_id = $ratissage->id;
        //     $dossier->save();
        // }

        return Response::json(['success' => 1], 201);
    }



    public function file(string $file_id)
    {
        $document = RatissageDocument::where('id', $file_id)->get()->first();
        if (is_null($document)) {
            abort(404);
        } else {
            $headers = array(
                'Content-Type: application/pdf',
            );
            return Response::download($document->documentPath(), $document->filename, $headers, "inline");
        }
    }
    public function file_data(string $file_id)
    {
        $dossier_document = BaraqueDocument::where('id', $file_id)->get()->first();
        return $dossier_document;
    }


    public function uploadMenageDocument(Request $request)
    {
        $now = Carbon::now();
        $now->hour(23);
        $now->minute(59);
        $now->second(59);

        $date_seisme = Carbon::createFromFormat('d/m/Y', "08/09/2023");
        $date_seisme->hour(00);
        $date_seisme->minute(00);
        $date_seisme->second(00);

        $menage_document = new MenageDocument();

        if ($request->id != null) {
            $menage_document = MenageDocument::where('id', $request->id)->first();
        } else {
            $menage_document->id = Str::uuid();
        }

        $file = $request->file('file');

        if ($file == null) {
            return Response::json(['success' => 0, 'error' => 'NO_FILE']);
        }

        $fileName = $file->getClientOriginalName();
        $file_extension = Str::upper($file->getClientOriginalExtension());
        $file_size = $file->getSize();

        $menage_document->filename = $fileName;
        $menage_document->nom = $fileName;
        $menage_document->user_id = Auth::user()->id;

        // Set menage_id, chef_menage_id, or commerce_id
        if ($request->commerce_id) {
            // For commerce documents
            $menage_document->commerce_id = $request->commerce_id;
            $commerce = Commerce::find($request->commerce_id);
            if ($commerce && $commerce->ratissage_id) {
                // Get menage_id from ratissage's first menage (or handle differently)
                $ratissage = Ratissage::find($commerce->ratissage_id);
                if ($ratissage && $ratissage->menages->count() > 0) {
                    $menage_document->menage_id = $ratissage->menages->first()->id;
                }
            }
        } elseif ($request->menage_id) {
            $menage_document->menage_id = $request->menage_id;
        } elseif ($request->chef_menage_id) {
            $chef_menage = ChefMenage::find($request->chef_menage_id);
            if ($chef_menage && $chef_menage->menage_id) {
                $menage_document->menage_id = $chef_menage->menage_id;
            }
        }

        if ($request->chef_menage_id) {
            $menage_document->chef_menage_id = $request->chef_menage_id;
        }

        // Validate that menage_id is set (required field)
        if (!$menage_document->menage_id) {
            return Response::json(['success' => 0, 'error' => 'MENAGE_ID_REQUIRED']);
        }

        $check_for_date_doc = 1;

        // Documents that don't require date validation
        if (
            $request->type_document == "PLAN_COTE_PDF"
            || $request->type_document == "ARRETE_DEMOLITION"
            || $request->type_document == "COMMERCE_ATTESTATION_DEMOLITION"
            || strpos($request->type_document, '_PHOTO') !== false
        ) {
            $check_for_date_doc = 0;
        }

        if ($check_for_date_doc == 1) {
            if ($request->date_document == null) {
                return Response::json(['success' => 0, 'error' => 'EMPTY_DATE_DOCUMENT']);
            } else {
                $date_doc = Carbon::createFromFormat('d/m/Y', $request->date_document);
                $date_doc->hour(00);
                $date_doc->minute(00);
                $date_doc->second(00);

                if ($date_doc->gt($now)) {
                    return Response::json(['success' => 0, 'error' => 'DATE_IN_FUTURE']);
                }

                if ($date_doc->lt($date_seisme)) {
                    return Response::json(['success' => 0, 'error' => 'DATE_LT_DATE_SEISME']);
                }

                $menage_document->date_document = $date_doc->format('Y-m-d');
            }
        } else {
            $menage_document->date_document = $now->format('Y-m-d');
        }

        $menage_document->file_type = $file_extension;
        $menage_document->file_size = $file_size;
        $menage_document->type_document = $request->type_document;
        $menage_document->active = 1;
        $menage_document->is_duplicate = 0;
        $menage_document->decision = 'En attente';
        $menage_document->save();

        if ($request->date_document != null) {
            $menage_document->formated_date_document = $menage_document->formated_date_document();
        }

        $menage_document->formated_created_at = $menage_document->formated_created_at();

        // Ensure directory exists
        $uploadPath = "files/menages_documents/" . $menage_document->id . "/";
        if (!file_exists($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        $file->move($uploadPath, $fileName);

        // Create thumbnail for specific documents if needed
        $create_thumbnail = 0;
        if (
            $request->type_document == "CM_MENAGE_CIN"
            || $request->type_document == "ATTESTATION_DEMOLITION"
            || strpos($request->type_document, 'CM_CIN_CONJOINT') !== false
            || strpos($request->type_document, 'CM_ACT_MARIAGE') !== false
        ) {
            $create_thumbnail = 1;
        }

        if ($create_thumbnail == 1) {
            try {
                $pdf = new \Spatie\PdfToImage\Pdf($menage_document->documentPath());
                $pdf->format(\Spatie\PdfToImage\Enums\OutputFormat::Jpg)
                    ->selectPage(1)
                    ->resolution(150)
                    ->quality(90)
                    ->save($menage_document->documentThumbnailPath());
            } catch (\Exception $e) {
                \Log::error('Thumbnail creation failed: ' . $e->getMessage());
            }
        }

        return Response::json(['success' => 1, 'document' => $menage_document]);
    }

    public function downloadMenageDocument($id)
    {
        $document = MenageDocument::findOrFail($id);
        $filePath = $document->documentPath();

        if (!file_exists($filePath)) {
            abort(404, 'File not found');
        }

        return response()->download($filePath, $document->filename);
    }

    public function upload(Request $request)
    {
        $now = Carbon::now();
        $now->hour(23);
        $now->minute(59);
        $now->second(59);

        $date_seisme = Carbon::createFromFormat('d/m/Y', "08/09/2023");
        $date_seisme->hour(00);
        $date_seisme->minute(00);
        $date_seisme->second(00);

        $ratissage_document = new RatissageDocument();

        if ($request->id != null) {
            $ratissage_document = RatissageDocument::where('id', $request->id)->get()->first();
        } else {
            $ratissage_document->id = Str::uuid();
        }

        $file = $request->file('file');

        if ($file == null) {
            if ($request->type_document == "RECEPISSE") {
                $fileName = "RECEPISSE.pdf";
                $file_extension = "PDF";
                $file_size = 0;
            }
        } else {
            $fileName = $file->getClientOriginalName();
            $file_extension = Str::upper($file->getClientOriginalExtension());
            $file_size = $file->getSize();
        }


        $ratissage_document->filename = $fileName;
        $ratissage_document->nom =  $fileName;
        $ratissage_document->user_id =  Auth::user()->id;

        $check_for_date_doc = 1;
        if (
            $request->type_document == "PLAN_COTE_PDF"
            || $request->type_document == "ARRETE_DEMOLITION"
            || strpos($request->type_document, '_PHOTO') !== false

        ) {

            $check_for_date_doc = 0;
        }

        if ($check_for_date_doc == 1) {
            if ($request->date_document == null) {
                return Response::json(['success' => 0, 'error' => 'EMPTY_DATE_DOCUMENT']);
            } else {

                $date_doc = Carbon::createFromFormat('d/m/Y', $request->date_document);
                $date_doc->hour(00);
                $date_doc->minute(00);
                $date_doc->second(00);

                if ($date_doc->gt($now)) {
                    return Response::json(['success' => 0, 'error' => 'DATE_IN_FUTURE']);
                }
                if ($date_doc->lt($date_seisme)) {
                    return Response::json(['success' => 0, 'error' => 'DATE_LT_DATE_SEISME']);
                }
                $ratissage_document->date_document = $date_doc->format('Y-m-d');
            }
        } else {
            $ratissage_document->date_document = $now->format('Y-m-d');
        }

        $ratissage_document->file_type = $file_extension;
        $ratissage_document->file_size = $file_size;

        $ratissage_document->type_document = $request->type_document;
        $ratissage_document->save();

        $create_thumbnail = 0;
        if (
            $request->type_document == "ARRETE_DEMOLITION"
            ||  $request->type_document == "PV_ETUDE"
            ||  $request->type_document == "PV_TAS"
        ) {
            $create_thumbnail = 1;
            // self::update_dossier_etat($request->menage_id);
        }

        if ($request->date_document != null) {
            $ratissage_document->formated_date_document = $ratissage_document->formated_date_document();
        }
        $ratissage_document->formated_created_at = $ratissage_document->formated_created_at();
        $file->move("files/ratissages_docs/" . $ratissage_document->id . "/", $fileName);


        if ($create_thumbnail == 1) {
            if ($create_thumbnail == 1) {
                $pdf = new Pdf($ratissage_document->documentPath());

                $pdf->format(\Spatie\PdfToImage\Enums\OutputFormat::Jpg)
                    ->selectPage(1)
                    ->resolution(150)
                    ->quality(90)
                    ->save($ratissage_document->documentThumbnailPath());
            }



            // $resolution = 150;
            // $width = 500;

            // $pdfPathEscaped = escapeshellarg("{$ratissage_document->documentPath()}[0]");
            // $outputJpgEscaped = escapeshellarg($ratissage_document->documentThumbnailPath());

            // $cmd = "convert -density $resolution $pdfPathEscaped -background white -alpha remove -alpha off -quality 90 -resize {$width} $outputJpgEscaped";

            // exec($cmd, $output, $return_var);
        }
        return Response::json(['success' => 1, 'document' => $ratissage_document]);
    }



    public function check_permission($baraque)
    {
        if (Auth::user()->has_role('Administrateur')) {
            // if ($dossier->filiale_id != Auth::user()->filiale_id) {
            //     abort(404);
            // }
        }
    }


    public function validate_document(Request $request)
    {
        // if (Auth::user()->has_role('Vérificateur,Administrateur')) {
        $dosisser_document = BaraqueDocument::where('id', $request->id)->get()->first();
        $dosisser_document->decision = 'Validé';
        $dosisser_document->verified_by = Auth::user()->id;
        $dosisser_document->verified_at = Carbon::now();
        $dosisser_document->save();
        $this->update_dossier_etat($dosisser_document->menage_id);
        return Response::json(['success' => 1, 'error' => '', 'menage_id' => $dosisser_document->menage_id]);
        // } else {
        //     return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED']);
        // }
    }
    public function rejeter_document(Request $request)
    {
        // if (Auth::user()->has_role('Vérificateur,Administrateur')) {
        $dosisser_document = BaraqueDocument::where('id', $request->id)->get()->first();
        $dosisser_document->decision = 'Rejeté';
        $dosisser_document->verified_by = Auth::user()->id;
        $dosisser_document->verified_at = Carbon::now();
        $dosisser_document->motif_rejet =  $request->motif_rejet;
        $dosisser_document->save();
        $this->update_dossier_etat($dosisser_document->menage_id);
        return Response::json(['success' => 1, 'error' => '']);
        // } else {
        //     return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED']);
        // }
    }
}
