<?php

namespace App\Http\Controllers;

use App\Models\Affectation;
use App\Models\Cercle;
use App\Models\Commune;
use App\Models\Douar;
use App\Models\Deblaiement;
use App\Models\Etude_geotechnique;
use App\Models\Etude_geotechnique_conclusion;
use App\Models\Etude_geotechnique_document;
use App\Models\Province;
use App\Models\Plan_type;
use App\Models\Plan_type_remarque;
use App\Models\Dossier_document;
use App\Models\Petitionnaire;
use App\Models\LPetitionnaire;
use App\Models\Petitionnaire_dr;
use App\Models\LPetitionnaire_dr;
use App\Models\DossierRehab_document;
use App\Models\Demande_document;
use App\Models\Demande;
use App\Models\Dossier;
use Illuminate\Support\Facades\DB;
use App\Models\Role;
use App\Models\User;
use App\Models\Guichet;
use AWS\CRT\HTTP\Response as HTTPResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\File;
use DataTables;
use Response;
use stdClass;
use Illuminate\Support\Facades\Storage;
use Illuminate\Filesystem\Filesystem;


include(app_path() . '/Http/Controllers/geoPHP.inc');

class CentreDonneesController extends Controller
{
    public function index()
    {
          

        if (Auth::user()->has_role('Observateur,Administrateur,Chef du projet')) {
            $douars = Auth::user()->douars();
            $communes = Auth::user()->communes();
            $cercles = Auth::user()->cercles();
            $provinces = Auth::user()->provinces();

            $role_prestataires = Role::whereIn('name', ['Laboratoire', 'Architecte', 'IGT', 'BET', 'Bureau de contrôle', 'Équipement'])->get();
            $prestataires = User::where('role_id', $role_prestataires->first()->id)->get();


            for ($i = 0; $i < count($douars); $i++) {
                if (strlen($douars[$i]->nom) == 0) {
                    $douars[$i]->nom = $douars[$i]->nom_ar;
                }
            }

            return view('centre_donnees.index', [

                'role_prestataires' => $role_prestataires,
                'prestataires' => $prestataires,

                'douars' => $douars,
                'communes' => $communes,
                'cercles' => $cercles,
                'provinces' => $provinces
            ]);
        } else {
            abort(403, 'Unauthorized action.');
        }
    }

    public function guichets_get(string $id)
    {
        $guichet = Guichet::where('id', $id)->get()->first();

        if ($guichet->date_creation != null) {
            $guichet->date_creation_fromated = Carbon::createFromFormat('Y-m-d',  $guichet->date_creation)->format('d/m/Y');
        }
        return  $guichet;
    }
    public function guichets_store(Request $request)
    {

        $guichet = Guichet::where('id', $request->id)->get()->first();

        if ($request->creation != null) {
            $guichet->creation = 1;
            $guichet->date_creation = Carbon::createFromFormat('d/m/Y',  $request->date_creation)->format('Y-m-d');
        } else {
            $guichet->creation = 0;
            $guichet->date_creation = null;
        }

        $charges_guichets_ids = json_decode($request->charges_guichets_ids);

        foreach ($charges_guichets_ids as  $charge_guichet_id) {
            $user = User::where('id', $charge_guichet_id)->get()->first();
            $user->guichet_id =  $guichet->id;
            $user->save();
        }


        $guichet->save();

        return Response::json(['success' => 0], 201);
    }

    public function guichets_datatable()
    {
        if (Auth::user()->has_role("Observateur de suivi")) {
            $guichets = Guichet::select('guichets.*')->join('provinces', 'provinces.id', '=', 'guichets.province_id')->get();
        } else {
            $guichets = Guichet::select('guichets.*')->join('provinces', 'provinces.id', '=', 'guichets.province_id')->where('provinces.filiale_id', Auth::user()->filiale_id)->get();
        }


        return Datatables::of($guichets)
            ->editColumn('nom', function ($guichet) {
                return  $guichet->nom;
            })
            ->editColumn('province', function ($guichet) {
                return $guichet->province->nom;
            })
            ->editColumn('creation', function ($guichet) {

                if (Auth::user()->has_role('Agence urbaine')) {
                    if ($guichet->creation == 1) {
                        return '<a class="guichet_a" item_id="' . $guichet->id . '" href="javascript:;"><b style="color: #5fa12c; font-weight: 500;">Oui</b></a>';
                    } else {
                        return '<a class="guichet_a" item_id="' . $guichet->id . '" href="javascript:;"><b style="color:gray">Non</b></a>';
                    }
                } else {
                    if ($guichet->creation == 1) {
                        return '<b style="color: #5fa12c; font-weight: 500;">Oui</b>';
                    } else {
                        return '<b style="color:gray">Non</b>';
                    }
                }
            })
            ->editColumn('date_creation', function ($guichet) {
                if ($guichet->date_creation != null) {
                    return '<b style="color: #5fa12c; font-weight: 500;">' . Carbon::createFromFormat('Y-m-d', $guichet->date_creation)->format('d/m/Y') . '</b>';
                } else {
                    return '<b style="color:gray">Néant</b>';
                }
            })
            ->rawColumns(['nom', 'province', 'creation', 'date_creation'])
            ->make(true);
    }
    public function douars_datatable(Request $request)
    {
        $filters = json_decode($request->filters);
        $douars = Douar::with('plan_types')->where('active', 1)->select('douars.*');

        foreach ($filters as $filter) {
            if ($filter->column == 'etude_geotechnique') {
                if ($filter->value == 1) {
                    $douars = $douars->whereNotNull('etude_geotechnique_id');
                } else {
                    $douars = $douars->whereNull('etude_geotechnique_id');
                }
                continue;
            }
            if ($filter->column == 'plan_type') {
                if ($filter->value == 1) {
                    $douars = $douars->join('plan_types', 'plan_types.douar_id', '=', 'douars.id');
                } else {
                    $douars = $douars->join('plan_types', 'plan_types.douar_id', '=', 'douars.id', 'left')->whereNull('plan_types.douar_id');
                }
                continue;
            }
            if ($filter->column == 'created_at') {

                $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
                $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));

                $douars = $douars->whereBetween($filter->column, [$from, $to]);
            } else {
                if ($filter->table == '') {
                    if ($filter->column == 'etat_deblaiement_l') {
                        if (str_starts_with($filter->value, 'Entamé')) {
                            $douars = $douars->where($filter->column, 'LIKE',  "%{$filter->value}%");
                        } else {
                            $douars = $douars->where($filter->column, '=', $filter->value);
                        }
                    } else {
                        $douars = $douars->where($filter->column, '=', $filter->value);
                    }
                } else {


                    $douars = $douars->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, '=', $filter->value);
                    });
                }
            }
        }

        if (Auth::user()->role->name == "Architecte") {

            $douars = $douars->whereHas('affectations', function ($query) {
                $query->where([['affectations.user_id',  Auth::user()->id]]);
            });
        }
        if (Auth::user()->role->name == "Laboratoire") {

            $douars = $douars->whereHas('affectations', function ($query) {
                $query->where([['affectations.user_id',  Auth::user()->id]]);
            });
        }

        if (Auth::user()->role->name == "BET") {

            $douars = $douars->whereHas('affectations', function ($query) {
                $query->where([['affectations.user_id',  Auth::user()->id]]);
            });
        }

        if (Auth::user()->role->name == "IGT") {

            $douars = $douars->whereHas('affectations', function ($query) {
                $query->where([['affectations.user_id',  Auth::user()->id]]);
            });
        }
        if (Auth::user()->role->name == "Administrateur" || Auth::user()->role->name == "Chef du projet") {
            $douars->where('douars.filiale_id', Auth::user()->filiale_id);
        }

        if (Auth::user()->role->name == "Agence urbaine") {
            $douars->where('douars.filiale_id', Auth::user()->filiale_id);
        }

        if (Auth::user()->role->name == "Observateur filiale") {
            $douars->where('douars.filiale_id', Auth::user()->filiale_id);
            if (!is_null(Auth::user()->province_id)) {
                $douars =  $douars->where('douars.province_id', Auth::user()->province_id);
            }
        }
        $start = request('start');
        $limit = request('length');

        $total_count = $douars->count();

        $douars = $douars->orderBy('nom_ar', 'asc')->offset($start)->limit($limit);

        $douars = $douars->groupBy('id');

        $douars = $douars->get();

        $json = Datatables::of($douars)
            ->editColumn('douar', function ($douar) {
                return '<input style="vertical-align: middle;position: relative;" class="index_douar_cb" item_id="' . $douar->id . '" type="checkbox">&nbsp' . (strlen($douar->nom) > 0 ? $douar->nom : $douar->nom_ar) . '';
            })
            ->editColumn('douar_cb', function ($douar) {

                return '<input style="vertical-align: middle;position: relative;" class="douar_cb" item_id="' . $douar->id . '" type="checkbox">&nbsp' . (strlen($douar->nom) > 0 ? $douar->nom : $douar->nom_ar) . '';
            })
            ->addColumn('province', function ($douar) {

                return $douar->commune->cercle->province->nom;
            })
            ->addColumn('cercle', function ($douar) {

                return $douar->commune->cercle->nom;
            })
            ->addColumn('commune', function ($douar) {

                return $douar->commune->nom;
            })
            ->addColumn('laboratoire', function ($douar) {

                $count = 0;
                $html = '<select class="form-control select2nc">';
                foreach ($douar->affectations as $affectation) {
                    if ($affectation->type == "Laboratoire") {
                        $html .= '<option>' . $affectation->user->organisme_ste . '</option>';
                        $count++;
                    }
                }
                $html .= '</select>';
                if ($count == 0) {
                    return '<b style="color:gray">Néant</b>';
                } else {
                    return $html;
                }
            })
            ->addColumn('architecte', function ($douar) {

                $count = 0;
                $html = '<select class="form-control select2nc">';
                foreach ($douar->affectations as $affectation) {
                    if ($affectation->type == "Architecte") {
                        $html .= '<option>' . $affectation->user->organisme_ste . '</option>';
                        $count++;
                    }
                }
                $html .= '</select>';
                if ($count == 0) {
                    return '<b style="color:gray">Néant</b>';
                } else {
                    return $html;
                }
            })
            ->addColumn('igt', function ($douar) {

                $count = 0;
                $html = '<select class="form-control select2nc">';
                foreach ($douar->affectations as $affectation) {
                    if ($affectation->type == "IGT") {
                        $html .= '<option>' . $affectation->user->organisme_ste . '</option>';
                        $count++;
                    }
                }
                $html .= '</select>';
                if ($count == 0) {
                    return '<b style="color:gray">Néant</b>';
                } else {
                    return $html;
                }
            })
            ->addColumn('bet', function ($douar) {

                $count = 0;
                $html = '<select class="form-control select2nc">';
                foreach ($douar->affectations as $affectation) {
                    if ($affectation->type == "BET") {
                        $html .= '<option>' . $affectation->user->organisme_ste . '</option>';
                        $count++;
                    }
                }
                $html .= '</select>';
                if ($count == 0) {
                    return '<b style="color:gray">Néant</b>';
                } else {
                    return $html;
                }
            })
            ->addColumn('bc', function ($douar) {

                $count = 0;
                $html = '<select class="form-control select2nc">';
                foreach ($douar->affectations as $affectation) {
                    if ($affectation->type == "Bureau de contrôle") {
                        $html .= '<option>' . $affectation->user->organisme_ste . '</option>';
                        $count++;
                    }
                }
                $html .= '</select>';
                if ($count == 0) {
                    return '<b style="color:gray">Néant</b>';
                } else {
                    return $html;
                }
            })
            ->addColumn('etude_geotechnique', function ($douar) {

                if ($douar->etude_geotechnique != null) {
                    $color = "#5fa12c";
                    if ($douar->etude_geotechnique->conclusion() == 'Non constructible') {
                        $color = "#F44141";
                    }
                    if ($douar->etude_geotechnique->conclusion() == 'Constructible avec réserves') {
                        $color = "#F28B1A";
                    }
                    if ($douar->etude_geotechnique->conclusion() == 'Mixte') {
                        $color = "#F28B1A";
                    }
                    if (Auth::user()->has_role('Laboratoire,Observateur de suivi')) {
                        return '<a style="color:' . $color . ' !important" class="etude_geotechnique_a" href="javascript:;" item_id="' . $douar->id . '">' . $douar->etude_geotechnique->conclusion() . '</a>';
                    } else {
                        return '<a style="color:' . $color . ' !important"   item_id="' . $douar->id . '">' . $douar->etude_geotechnique->conclusion() . '</a>';
                    }
                } else {
                    if (Auth::user()->has_role('Laboratoire')) {
                        return '<a class="etude_geotechnique_a" href="javascript:;" item_id="' . $douar->id . '">Non établi</a>';
                    } else {
                        return '<a item_id="' . $douar->id . '">Non établi</a>';
                    }
                }
            })
            ->addColumn('deblaiement', function ($douar) {

                $deblaiement =  $douar->deblaiement;
                $color = "#000000";
                if (str_starts_with($deblaiement, 'Non concerné')) {
                    $color = "#808080";
                }
                if (str_starts_with($deblaiement, 'Concerné')) {
                    $color = "#0F91C8";
                }

                if (Auth::user()->has_role('Équipement')) {
                    return '<a href="javascript:;"  douar_id="' . $douar->id . '"  class="deblaiement_a"><b style="color:' . $color . '; font-weight: 500;">' . $deblaiement . '</b></a>';
                } else {
                    return ' <b style="color:' . $color . '; font-weight: 500;">' . $deblaiement . '</b>';
                }
            })

            ->addColumn('etat_deblaiement', function ($douar) {

                $etat_deblaiement =  $douar->etat_deblaiement_l;
                $color = "#000000";

                if (str_starts_with($etat_deblaiement, 'Entamé')) {
                    $color = "#F08A38";
                }
                if (str_starts_with($etat_deblaiement, 'Achevé')) {
                    $color = "#79A12C";
                }

                if (Auth::user()->has_role('Équipement') && $douar->deblaiement == "Concerné") {
                    return '<a href="javascript:;"  douar_id="' . $douar->id . '"  class="etat_deblaiement_a"><b style="color:' . $color . '; font-weight: 500;">' . $etat_deblaiement . '</b></a>';
                } else {
                    return ' <b style="color:' . $color . '; font-weight: 500;">' . $etat_deblaiement . '</b>';
                }
            })
            ->addColumn('stabilisation_sol', function ($douar) {

                if ($douar->stabilisation_sol == 1) {
                    return '<b style="color: #5fa12c; font-weight: 500;">Oui</b>';
                } else {
                    return '<b style="color:gray">Non</b>';
                }
            })

            ->addColumn('nivellement', function ($douar) {
                if ($douar->nivellement == 1) {
                    return '<b style="color: #5fa12c; font-weight: 500;">Oui</b>';
                } else {
                    return '<b style="color:gray">Non</b>';
                }
            })
            ->addColumn('prestataire', function ($douar)  use ($request) {

                if ($request->prestataire_role != null) {
                    $count = 0;
                    $html = '<select class="form-control select2nc">';
                    foreach ($douar->affectations as $affectation) {
                        if ($affectation->user->role_id == $request->prestataire_role) {
                            $html .= '<option>' . $affectation->user->organisme_ste . '</option>';
                            $count++;
                        }
                    }
                    $html .= '</select>';
                    if ($count == 0) {
                        return '<b style="color:gray">Néant</b>';
                    } else {
                        return $html;
                    }
                }
                // if ($request->prestataire_role != null) {
                //     if ($douar->laboratoire != null) {
                //         if ($douar->laboratoire->role_id == $request->prestataire_role) {
                //             return '<b>' . $douar->laboratoire->organisme_ste . '</b>';
                //         }
                //     }
                //     if ($douar->igt != null) {
                //         if ($douar->igt->role_id == $request->prestataire_role) {
                //             return '<b>' . $douar->igt->organisme_ste . '</b>';
                //         }
                //     }

                //     if ($douar->bet != null) {
                //         if ($douar->bet->role_id == $request->prestataire_role) {
                //             return '<b>' . $douar->bet->organisme_ste . '</b>';
                //         }
                //     }

                //     if ($douar->bc != null) {
                //         if ($douar->bc->role_id == $request->prestataire_role) {
                //             return '<b>' . $douar->bc->organisme_ste . '</b>';
                //         }
                //     }
                //     if ($douar->architecte != null) {
                //         if ($douar->architecte->role_id == $request->prestataire_role) {
                //             return '<b>' . $douar->architecte->organisme_ste . '</b>';
                //         }
                //     }
                // }
            })
            ->addColumn('plans_type', function ($douar) {

                $plans_type_architecte_count = 0;

                foreach ($douar->plans_type as $plan_type) {
                    if ($plan_type->type == "PLAN_ARCHITECTE") {
                        $plans_type_architecte_count++;
                    }
                }

                if ($plans_type_architecte_count == 0) {
                    $color = "gray";
                    return '<a target="_blank" href="/centre_donnees/plan_types/' . $douar->id . '" style="color:' . $color . ' !important">Néant</b>';
                } else {

                    $color = "#5fa12c";
                    foreach ($douar->plans_type as $plan_type) {
                        if ($plan_type->type == "PLAN_ARCHITECTE") {
                            if ($plan_type->etat == 'En cours') {
                                $color = "#FF8040";
                            }
                        }
                    }
                    return '<a target="_blank" href="/centre_donnees/plan_types/' . $douar->id . '" style="color:' . $color . ' !important">' . $plans_type_architecte_count . ' fichier(s)</b>';
                }
            })
            ->addColumn('plans_type_bet', function ($douar) {


                $plans_type_architecte_count = 0;

                foreach ($douar->plans_type as $plan_type) {
                    if ($plan_type->type == "BET") {
                        $plans_type_architecte_count++;
                    }
                }

                if ($plans_type_architecte_count == 0) {
                    $color = "gray";
                    return '<a target="_blank" href="/centre_donnees/plan_types_bet/' . $douar->id . '" style="color:' . $color . ' !important">Néant</b>';
                } else {

                    $color = "#5fa12c";
                    foreach ($douar->plans_type as $plan_type) {
                        if ($plan_type->type == "BET") {
                            if ($plan_type->etat == 'En cours') {
                                $color = "#FF8040";
                            }
                        }
                    }
                    return '<a target="_blank" href="/centre_donnees/plan_types_bet/' . $douar->id . '" style="color:' . $color . ' !important">' . $plans_type_architecte_count . ' fichier(s)</b>';
                }
            })

            ->rawColumns(['plans_type_bet', 'plans_type', 'douar', 'nivellement', 'stabilisation_sol', 'etat_deblaiement', 'deblaiement', 'etude_geotechnique', 'bc', 'bet', 'igt', 'architecte', 'prestataire', 'laboratoire', 'douar_cb', 'numero', 'petitionnaires', 'rapport_geotechnique', 'etat_avancement', 'aide_frontale', 'permis_habiter', 'date'])
            ->skipPaging()->setTotalRecords($total_count)->setFilteredRecords($total_count)->tojson();



        return $json;
    }

    public function  douar(String $id)
    {
        $douar = Douar::with('etude_geotechnique.etudes_geotechniques_conclusions', 'etude_geotechnique.etudes_geotechniques_documents')->where('id', $id)->get()->first();

        if ($douar != null) {
            if ($douar->etude_geotechnique != null) {
                $douar->etude_geotechnique->created_at_formated = Carbon::createFromFormat('Y-m-d H:i:s', $douar->etude_geotechnique->created_at)->format('d/m/Y');

                $douar->etude_geotechnique->conclusion = $douar->etude_geotechnique->conclusion();

                foreach ($douar->etude_geotechnique->etudes_geotechniques_documents as $etudes_geotechniques_document) {
                    $etudes_geotechniques_document->created_at_formated = Carbon::createFromFormat('Y-m-d H:i:s', $etudes_geotechniques_document->created_at)->format('d/m/Y');
                }
            }
            return  $douar;
        } else {
            return Response::json(['success' => 0, 'error' => 'UNKNOWN_DOUAR'], 200);
        }
    }

    public function  douars_edit_etat_sol(Request $request)
    {

        Douar::whereIn('id', explode(',', $request->ids))->update(array('deblaiement' => ($request->deblaiement != null ? 1 : 0)));
        Douar::whereIn('id', explode(',', $request->ids))->update(array('stabilisation_sol' => ($request->stabilisation_sol != null ? 1 : 0)));
        Douar::whereIn('id', explode(',', $request->ids))->update(array('nivellement' => ($request->nivellement != null ? 1 : 0)));
        return Response::json(['success' => 1], 200);
    }

    public function prestataires_datatable(Request $request)
    {
        $filters = json_decode($request->filters);
        $prestataires = User::with('role')->where([['active', 1]]);

        $prestataires = $prestataires->whereHas('role', function ($query) {
            $query->where('roles' . '.' . 'prestataire', '=', 1);
        });

        foreach ($filters as $filter) {
            if ($filter->column == 'created_at') {

                $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
                $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));

                $prestataires = $prestataires->whereBetween($filter->column, [$from, $to]);
            } else {
                if ($filter->table == '') {
                    $prestataires = $douars->where($filter->column, '=', $filter->value);
                } else {
                    $prestataires = $prestataires->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, '=', $filter->value);
                    });
                }
            }
        }
        $prestataires = $prestataires->orderBy('created_at', 'desc')->get();


        if (Auth::user()->has_role('Agence urbaine')) {
            $prestataires = $prestataires->where('role_id', '8c2d8e8f-8a54-47a9-bcc1-3dc0c6255220');
        }
        return Datatables::of($prestataires)
            ->addColumn('email', function ($prestataire) {
                return '<input style="vertical-align: middle;position: relative;" class="prestataire_cb" item_id="' . $prestataire->id . '" type="checkbox">&nbsp<a class="prestataire_a" item_id="' . $prestataire->id . '" href="javascript:;" >' . $prestataire->email . '</a>';
            })
            ->addColumn('nom', function ($prestataire) {
                return $prestataire->nom;
            })
            ->addColumn('prenom', function ($prestataire) {
                return $prestataire->prenom;
            })
            ->addColumn('tel', function ($prestataire) {
                return $prestataire->tel;
            })
            ->addColumn('organisme_ste', function ($prestataire) {
                return $prestataire->organisme_ste;
            })
            ->addColumn('type', function ($prestataire) {
                return $prestataire->role->name;
            })
            ->addColumn('date', function ($prestataire) {

                return Carbon::createFromFormat('Y-m-d H:i:s', $prestataire->created_at)->format('d/m/Y H:i');
            })

            ->addColumn('etat_engagement', function ($prestataire) {

                $etat_engagements = $prestataire->etat_engagements;
                if (count($etat_engagements) > 0) {
                    $html = "";
                    foreach ($etat_engagements as $etat_engagement) {
                        if ($etat_engagement->filiale_id == Auth::user()->filiale_id) {
                            $html .= '<a class="etat_engagement_a" user_id="' . $etat_engagement->user_id . '" href="javascript:;" style="color:#79A12C !important">•&nbsp<b style="color:#79A12C">' . $etat_engagement->filiale->nom . '</b> - ' . $etat_engagement->etat . '</a></br>';
                        } else {
                            $html .= '<b style="color:#000 !important">•&nbsp<b style="color:#000">' . $etat_engagement->filiale->nom . '</b> - ' . $etat_engagement->etat . '</b></br>';
                        }
                    }
                    return $html;
                } else {
                    return '<b style="color:#999999">Néant</b>';
                }
            })
            ->addColumn('affecte', function ($prestataire) {

                if ($prestataire->affecte()) {
                    return '<b style="color:#79A12C">Oui</b>';
                } else {
                    return '<b style="color:#D56565">Non</b>';
                }
            })
            ->addColumn('invitation', function ($prestataire) {
                return  $prestataire->invitation_status();
            })


            ->rawColumns(['etat_engagement', 'affecte', 'invitation', 'email'])
            ->make(true);
    }

    public function charges_guichets_datatable(Request $request)
    {
        $filters = json_decode($request->filters);
        $prestataires = User::with('role')->where([['active', 1]]);


        foreach ($filters as $filter) {
            if ($filter->column == 'created_at') {

                $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
                $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));

                $prestataires = $prestataires->whereBetween($filter->column, [$from, $to]);
            } else {
                if ($filter->table == '') {
                    $prestataires = $douars->where($filter->column, '=', $filter->value);
                } else {
                    $prestataires = $prestataires->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, '=', $filter->value);
                    });
                }
            }
        }
        $prestataires = $prestataires->orderBy('created_at', 'desc')->get();


        if (Auth::user()->has_role('Agence urbaine')) {
            $prestataires = $prestataires->where('role_id', '8c2d8e8f-8a54-47a9-bcc1-3dc0c6255220');
            $prestataires = $prestataires->where('agence_urbain_id',  Auth::user()->id);
        }
        return Datatables::of($prestataires)
            ->addColumn('email', function ($prestataire) {
                return '<input style="vertical-align: middle;position: relative;" class="prestataire_cb" item_id="' . $prestataire->id . '" type="checkbox">&nbsp<a class="prestataire_a" item_id="' . $prestataire->id . '" href="javascript:;" >' . $prestataire->email . '</a>';
            })
            ->addColumn('nom', function ($prestataire) {
                return $prestataire->nom;
            })
            ->addColumn('prenom', function ($prestataire) {
                return $prestataire->prenom;
            })
            ->addColumn('tel', function ($prestataire) {
                return $prestataire->tel;
            })
            ->addColumn('date', function ($prestataire) {

                if (is_null($prestataire->created_at)) {
                    return "-";
                } else {
                    return Carbon::createFromFormat('Y-m-d H:i:s', $prestataire->created_at)->format('d/m/Y H:i');
                }
            })
            ->addColumn('invitation', function ($prestataire) {
                return  $prestataire->invitation_status();
            })
            ->addColumn('guichet', function ($prestataire) {

                if ($prestataire->guichet != null) {
                    return '<b style="color:#79A12C">' . $prestataire->guichet->nom . '</b>';
                } else {
                    return  '<b style="color:#7D7D7D">Non assigné</b>';
                }
            })

            ->rawColumns(['guichet', 'etat_engagement', 'affecte', 'invitation', 'email'])
            ->make(true);
    }


    public function  douars_store_etude_geotechnique(Request $request)
    {
        $douar = Douar::where('id', $request->id)->get()->first();

        if ($douar->etude_geotechnique == null) {
            $etude_geotechnique = new Etude_geotechnique();
            $etude_geotechnique->id = Str::uuid();
            $etude_geotechnique->save();
            $douar->etude_geotechnique_id = $etude_geotechnique->id;
            $douar->save();
        }

        $conclusions = json_decode($request->conclusions);

        $conclusions_in_db = Etude_geotechnique_conclusion::where('etude_geotechnique_id', $douar->etude_geotechnique_id)->get();
        foreach ($conclusions_in_db as $conclusion_db) {
            $exists = false;
            foreach ($conclusions as $conclusion) {

                if ($conclusion->id == $conclusion_db->id) {
                    $exists = true;
                }
            }
            if (!$exists) {
                $conclusion_db->delete();
            }
        }

        foreach ($conclusions as $conclusion) {
            $etude_geotechnique_conclusion =  new Etude_geotechnique_conclusion();
            if ($conclusion->id == null) {
                $etude_geotechnique_conclusion->id = Str::uuid();
            } else {
                $etude_geotechnique_conclusion = Etude_geotechnique_conclusion::where('id', $conclusion->id)->get()->first();
            }

            $etude_geotechnique_conclusion->conclusion = $conclusion->conclusion;
            $etude_geotechnique_conclusion->commentaire = $conclusion->commentaire;
            $etude_geotechnique_conclusion->zone = $conclusion->zone;
            $etude_geotechnique_conclusion->etude_geotechnique_id =  $douar->etude_geotechnique_id;
            $etude_geotechnique_conclusion->save();
        }

        $rapoort_geotechnique_file = $request->file('file');

        if ($rapoort_geotechnique_file) {
            $fileName = $rapoort_geotechnique_file->getClientOriginalName();
            $rapoort_geotechnique_file->move("files/centre_donnees" . "/etude_geotechnique/" . $douar->etude_geotechnique_id . "/", $fileName);

            $etude_geotechnique_doc = new Etude_geotechnique_document();
            $etude_geotechnique_doc->etude_geotechnique_id = $douar->etude_geotechnique_id;
            $etude_geotechnique_doc->id = Str::uuid();
            $etude_geotechnique_doc->filename = $rapoort_geotechnique_file->getClientOriginalName();
            $etude_geotechnique_doc->file_type = $rapoort_geotechnique_file->getClientOriginalExtension();
            $etude_geotechnique_doc->file_size = 0;
            $etude_geotechnique_doc->type_document = 'RAPPORT_GEOTECHNIQUE';
            $etude_geotechnique_doc->save();
        }

        $rapoort_geotechnique_lpee_file = $request->file('file_lpee');

        if ($rapoort_geotechnique_lpee_file) {
            $fileName = $rapoort_geotechnique_lpee_file->getClientOriginalName();
            $rapoort_geotechnique_lpee_file->move("files/centre_donnees" . "/etude_geotechnique/" . $douar->etude_geotechnique_id . "/", $fileName);

            $etude_geotechnique_doc = new Etude_geotechnique_document();
            $etude_geotechnique_doc->etude_geotechnique_id = $douar->etude_geotechnique_id;
            $etude_geotechnique_doc->id = Str::uuid();
            $etude_geotechnique_doc->filename = $rapoort_geotechnique_lpee_file->getClientOriginalName();
            $etude_geotechnique_doc->file_type = $rapoort_geotechnique_lpee_file->getClientOriginalExtension();
            $etude_geotechnique_doc->file_size = 0;
            $etude_geotechnique_doc->type_document = 'RAPPORT_GEOTECHNIQUE_LPEE';
            $etude_geotechnique_doc->save();
        }


        $effet_topo_file = $request->file('effet_topo_file');

        if ($effet_topo_file) {
            $fileName = $effet_topo_file->getClientOriginalName();
            $effet_topo_file->move("files/centre_donnees" . "/etude_geotechnique/" . $douar->etude_geotechnique_id . "/", $fileName);

            $etude_geotechnique_doc = new Etude_geotechnique_document();
            $etude_geotechnique_doc->etude_geotechnique_id = $douar->etude_geotechnique_id;
            $etude_geotechnique_doc->id = Str::uuid();
            $etude_geotechnique_doc->filename = $effet_topo_file->getClientOriginalName();
            $etude_geotechnique_doc->file_type = $effet_topo_file->getClientOriginalExtension();
            $etude_geotechnique_doc->file_size = 0;
            $etude_geotechnique_doc->type_document = 'EFFET_TOPO';
            $etude_geotechnique_doc->save();
        }

        $effet_site_lithologique_file = $request->file('effet_site_lithologique_file');

        if ($effet_site_lithologique_file) {
            $fileName = $effet_site_lithologique_file->getClientOriginalName();
            $effet_site_lithologique_file->move("files/centre_donnees" . "/etude_geotechnique/" . $douar->etude_geotechnique_id . "/", $fileName);

            $etude_geotechnique_doc = new Etude_geotechnique_document();
            $etude_geotechnique_doc->etude_geotechnique_id = $douar->etude_geotechnique_id;
            $etude_geotechnique_doc->id = Str::uuid();
            $etude_geotechnique_doc->filename = $effet_site_lithologique_file->getClientOriginalName();
            $etude_geotechnique_doc->file_type = $effet_site_lithologique_file->getClientOriginalExtension();
            $etude_geotechnique_doc->file_size = 0;
            $etude_geotechnique_doc->type_document = 'EFFET_SITE_LITHOLOGIQUE';
            $etude_geotechnique_doc->save();
        }
        return Response::json(['success' => 1], 200);
    }

    public function  downloadEtudeGeotechnique($douar_id)
    {
        $auth_user = Auth::user();
        $role_name = $auth_user->role->name;

        $douar = Douar::with('etude_geotechnique')->where('id', $douar_id)->get()->first();

        if (!isset($douar->etude_geotechnique)) {
            abort(404);
        }
        $authorized = false;
        if ($role_name != "Administrateur" && $role_name != "Observateur de suivi" && $role_name != "Observateur filiale") {

            foreach ($douar->affectations as $affectation) {
                if ($affectation->user_id == $auth_user->id) {
                    $authorized = true;
                }
            }
        } else {
            $authorized = true;
        }

        if ($authorized == true) {
            $etudes_geotechnique = $douar->etude_geotechnique;
            if ($etudes_geotechnique != null) {

                foreach ($etudes_geotechnique->etudes_geotechniques_documents as $document) {
                    if ($document->type_document == "RAPPORT_GEOTECHNIQUE") {

                        $headers = array(
                            'Content-Type: application/pdf',
                        );
                        return Response::download($document->documentPath(), $document->filename, $headers);
                    }
                }
            } else {
                abort(404);
            }
        } else {
            abort(404);
        }
    }



    public function downloadEtudeGeotechniqueLPEE($douar_id)
    {
        $auth_user = Auth::user();
        $role_name = $auth_user->role->name;

        $douar = Douar::with('etude_geotechnique')->where('id', $douar_id)->get()->first();

        if (!isset($douar->etude_geotechnique)) {
            abort(404);
        }
        $authorized = false;
        if ($role_name != "Administrateur" && $role_name != "Observateur de suivi" && $role_name != "Observateur filiale") {

            foreach ($douar->affectations as $affectation) {
                if ($affectation->user_id == $auth_user->id) {
                    $authorized = true;
                }
            }
        } else {
            $authorized = true;
        }

        if ($authorized == true) {
            $etudes_geotechnique = $douar->etude_geotechnique;
            if ($etudes_geotechnique != null) {

                foreach ($etudes_geotechnique->etudes_geotechniques_documents as $document) {
                    if ($document->type_document == "RAPPORT_GEOTECHNIQUE_LPEE") {

                        $headers = array(
                            'Content-Type: application/pdf',
                        );
                        return Response::download($document->documentPath(), $document->filename, $headers);
                    }
                }
            } else {
                abort(404);
            }
        } else {
            abort(404);
        }
    }

    public function  downloadFileEffetLithologique($douar_id)
    {
        $auth_user = Auth::user();
        $role_name = $auth_user->role->name;

        $douar = Douar::with('etude_geotechnique')->where('id', $douar_id)->get()->first();

        if (!isset($douar->etude_geotechnique)) {
            abort(404);
        }
        $authorized = false;
        if ($role_name != "Administrateur" && $role_name != "Observateur de suivi" && $role_name != "Observateur filiale") {

            foreach ($douar->affectations as $affectation) {
                if ($affectation->user_id == $auth_user->id) {
                    $authorized = true;
                }
            }
        } else {
            $authorized = true;
        }

        if ($authorized == true) {
            $etudes_geotechnique = $douar->etude_geotechnique;
            if ($etudes_geotechnique != null) {

                foreach ($etudes_geotechnique->etudes_geotechniques_documents as $document) {
                    if ($document->type_document == "EFFET_SITE_LITHOLOGIQUE") {

                        $headers = array(
                            'Content-Type: application/geojson',
                        );
                        return Response::download($document->documentPath(), $document->filename, $headers);
                    }
                }
            } else {
                abort(404);
            }
        } else {
            abort(404);
        }
    }

    public function  downloadFileEffetTopo($douar_id)
    {
        $auth_user = Auth::user();
        $role_name = $auth_user->role->name;

        $douar = Douar::with('etude_geotechnique')->where('id', $douar_id)->get()->first();

        if (!isset($douar->etude_geotechnique)) {
            abort(404);
        }
        $authorized = false;
        if ($role_name != "Administrateur" && $role_name != "Observateur de suivi" && $role_name != "Observateur filiale") {
            foreach ($douar->affectations as $affectation) {
                if ($affectation->user_id == $auth_user->id) {
                    $authorized = true;
                }
            }
        } else {
            $authorized = true;
        }
        if ($authorized == true) {
            $etudes_geotechnique = $douar->etude_geotechnique;
            if ($etudes_geotechnique != null) {

                foreach ($etudes_geotechnique->etudes_geotechniques_documents as $document) {
                    if ($document->type_document == "EFFET_TOPO") {

                        $headers = array(
                            'Content-Type: application/geojson',
                        );
                        return Response::download($document->documentPath(), $document->filename, $headers);
                    }
                }
            } else {
                abort(404);
            }
        } else {
            abort(404);
        }
    }

    public function etude_geotechnique($etude_geotechnique_id)
    {
        $etude_geotechnique = Etude_geotechnique::with('etudes_geotechniques_conclusions', 'etudes_geotechniques_documents')->where('id', $etude_geotechnique_id)->get()->first();
        return  $etude_geotechnique;
    }

    public function conclusion($conclusion_id)
    {
        $etude_geotechnique_conclusion = Etude_geotechnique_conclusion::where('id', $conclusion_id)->get()->first();
        return  $etude_geotechnique_conclusion;
    }

    public function get_deblaiement(string $id)
    {
        $deblaiement = Deblaiement::where('douar_id', $id)->orderBy('created_at', 'desc')->get()->first();

        if (!is_null($deblaiement)) {
            if ($deblaiement->date_achevement != null) {
                $deblaiement->date_achevement_formated = Carbon::createFromFormat('Y-m-d',  $deblaiement->date_achevement)->format('d/m/Y');
            }
            return Response::json(['success' => 1, 'deblaiement' => $deblaiement], 200);
        } else {
            return Response::json(['success' => 0], 200);
        }
    }

    public function store_deblaiement(Request $request)
    {
        if (!Auth::user()->has_role('Équipement')) {
            return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED'], 200);
        }
        $douar = Douar::where('id', $request->douar_id)->get()->first();



        $deblaiement = new Deblaiement();
        $deblaiement->id = Str::uuid();
        $deblaiement->non_concerne = is_null($request->non_concerne) ? 0 : 1;
        $deblaiement->concerne = is_null($request->concerne) ? 0 : 1;
        $deblaiement->douar_id = $request->douar_id;
        $deblaiement->entame = is_null($request->entame) ? 0 : 1;
        $deblaiement->progression =  is_null($request->progression) ? 0 :  $request->progression;
        $deblaiement->acheve = is_null($request->acheve) ? 0 : 1;
        if (is_null($request->acheve)) {
            $deblaiement->date_achevement = null;
        } else {
            $deblaiement->date_achevement = Carbon::createFromFormat('d/m/Y',  $request->date_achevement);
        }

        if ($deblaiement->acheve == 1) {
            $deblaiement->progression = 100;
            $deblaiement->entame = 1;
        }

        $deblaiement->hash = hash('sha256', $deblaiement->non_concerne . ',' . $deblaiement->concerne . ',' . $deblaiement->douar_id . ',' . $deblaiement->entame . ',' . $deblaiement->progression . ',' . $deblaiement->acheve . "," . $request->date_achevement);

        if (Deblaiement::where([['douar_id', $deblaiement->douar_id], ['hash', $deblaiement->hash]])->count() > 0) {
            return Response::json(['success' => 1], 201);
        }
        $deblaiement->user_id = Auth::user()->id;
        $deblaiement->save();




        $douar->deblaiement = 'Non renseigné';
        if ($deblaiement->non_concerne == 1) {
            $douar->deblaiement = 'Non concerné';
        }
        if ($deblaiement->concerne == 1) {
            $douar->deblaiement = 'Concerné';
        }

        $douar->etat_deblaiement_l = 'Non renseigné';
        if ($deblaiement->entame == 1) {
            $douar->etat_deblaiement_l = 'Entamé ( ' . $deblaiement->progression . ' % )';
        }
        if ($deblaiement->acheve == 1) {
            $douar->etat_deblaiement_l = 'Achevé';
        }

        $douar->save();

        return Response::json(['success' => 1], 201);
    }

    public function deblaiements_datatable(Request $request)
    {
        $deblaiements = Deblaiement::where([['douar_id',  $request->douar_id]])->orderBy('created_at', 'asc')->get();

        return Datatables::of($deblaiements)

            ->addColumn('etat', function ($deblaiement) {

                return '<b><i class="fa-solid fa-check"></i>&nbsp ' . $deblaiement->etat() . '</b>';
            })
            ->addColumn('date', function ($deblaiement) {
                return Carbon::createFromFormat('Y-m-d H:i:s', $deblaiement->created_at)->format('d/m/Y');
            })
            ->addColumn('user', function ($deblaiement) {
                return User::where('id', $deblaiement->user_id)->get()->first()->organisme_ste;
            })

            ->rawColumns(['etat', 'date', 'user'])
            ->make(true);
    }

    public function get_douar_prestataires(string $douar_id)
    {
        $douar = Douar::where('id', $douar_id)->get()->first();
        $affectations = $douar->affectations;
        foreach ($affectations as $affectation) {
            $affectation->prestataire = User::select('id', 'nom', 'prenom', 'organisme_ste')->where('id', $affectation->user_id)->get()->first();
        }
        return  $affectations;
    }

    public function  plan_types_show(string $douar_id)
    {
        $douar = Douar::where('id', $douar_id)->get()->first();
        return view('centre_donnees.plan_types', [
            'douar' => $douar

        ]);
    }

    public function  plan_types_bet_show(string $douar_id)
    {
        $douar = Douar::where('id', $douar_id)->get()->first();
        return view('centre_donnees.plan_types_bet', [
            'douar' => $douar

        ]);
    }


    public function plan_types_bet_files_datatable(Request $request)
    {

        $plan_types = Plan_type::where([['douar_id',  $request->douar], ['type', 'BET']])->orderBy('created_at', 'asc');

        return Datatables::of($plan_types)

            ->addColumn('plan_type', function ($plan_type) {
                $cb = '';
                if (Auth::user()->has_role('BET')) {

                    $cb = '<input ' . ($plan_type->etat == 'Validé' ? 'disabled' : '') . ' style="vertical-align: middle;position: relative;" class="plant_type_cb" item_id="' .  $plan_type->id . '" type="checkbox">&nbsp';
                }
                return   $cb . '<a class="plant_type_a" item_id="' . $plan_type->id . '" href="javascript:;">Plan type ' . $plan_type->nom . '</a>';
            })
            ->addColumn('date', function ($plan_type) {
                return Carbon::createFromFormat('Y-m-d H:i:s', $plan_type->created_at)->format('d/m/Y');
            })
            ->addColumn('architecte', function ($plan_type) {
                return $plan_type->architecte->organisme_ste;
            })
            ->addColumn('etat', function ($plan_type) {
                return $plan_type->etat;
            })
            ->rawColumns(['plan_type', 'date', 'architecte', 'etat'])
            ->make(true);
    }

    public function plan_types_files_datatable(Request $request)
    {

        $plan_types = Plan_type::where([['douar_id',  $request->douar], ['type', 'PLAN_ARCHITECTE']])->orderBy('created_at', 'asc');

        return Datatables::of($plan_types)

            ->addColumn('plan_type', function ($plan_type) {
                $cb = '';
                if (Auth::user()->has_role('Architecte')) {

                    $cb = '<input ' . ($plan_type->etat == 'Validé' ? 'disabled' : '') . ' style="vertical-align: middle;position: relative;" class="plant_type_cb" item_id="' .  $plan_type->id . '" type="checkbox">&nbsp';
                }
                return   $cb . '<a class="plant_type_a" item_id="' . $plan_type->id . '" href="javascript:;">Plan type ' . $plan_type->nom . '</a>';
            })
            ->addColumn('date', function ($plan_type) {
                return Carbon::createFromFormat('Y-m-d H:i:s', $plan_type->created_at)->format('d/m/Y');
            })
            ->addColumn('architecte', function ($plan_type) {
                return $plan_type->architecte->organisme_ste;
            })
            ->addColumn('etat', function ($plan_type) {
                return $plan_type->etat;
            })
            ->rawColumns(['plan_type', 'date', 'architecte', 'etat'])
            ->make(true);
    }

    public function  plan_type_upload(Request $request)
    {
        $douar = Douar::where('id', $request->douar_id)->get()->first();
        if (Auth::user()->has_role('Architecte') || Auth::user()->has_role('BET')) {
            $plan_type = new Plan_type();
            $plan_type->id = Str::uuid();
            $plan_type->nom = $request->nom;
            $plan_type->douar_id = $request->douar_id;

            $plan_type->architecte_id = Auth::user()->id;
            $plan_type->filiale_id =    $douar->filiale_id;

            if (!is_null($request->type)) {
                $plan_type->type = $request->type;
            } else {
                $plan_type->type = "PLAN_ARCHITECTE";
            }

            $dwg_file = $request->file('dwg_file');

            if ($dwg_file) {
                $fileName = $dwg_file->getClientOriginalName();
                $dwg_file->move("files/centre_donnees" . "/plan_type/" . $plan_type->id . "/", $fileName);

                $plan_type->file_size_dwg = 0;
                $plan_type->file_type_dwg = "DWG";
                $plan_type->filename_dwg =  $fileName;
            }


            $pdf_file = $request->file('pdf_file');

            if ($pdf_file) {
                $fileName = $pdf_file->getClientOriginalName();
                $pdf_file->move("files/centre_donnees" . "/plan_type/" . $plan_type->id . "/", $fileName);

                $plan_type->file_size = 0;
                $plan_type->file_type = "PDF";
                $plan_type->filename =  $fileName;
            }




            $plan_type->save();

            return Response::json(['success' => 1], 201);
        } else {
            return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED'], 200);
        }
    }

    public function plan_type(string $plan_type_id)
    {
        $plan_type = Plan_type::with('architecte')->where([['id',  $plan_type_id]])->get()->first();
        $plan_type->created_at_formated = Carbon::createFromFormat('Y-m-d H:i:s', $plan_type->created_at)->format('d/m/Y');
        if (Auth::user()->has_role('Agence urbaine,Chargé du guichet d\'accompagnement')) {
            $plan_type->can_validate = true;
        } else {
            $plan_type->can_validate = false;
        }

        return $plan_type;
    }

    public function plan_type_file_pdf(string $plan_type_id)
    {
        $plan_type = Plan_type::where([['id',  $plan_type_id]])->get()->first();
        $headers = array(
            'Content-Type: application/pdf',
        );
        return Response::download($plan_type->documentPdfPath(), $plan_type->filename, $headers, 'inline');
    }

    public function plan_type_file_dwg(string $plan_type_id)
    {
        $plan_type = Plan_type::where([['id',  $plan_type_id]])->get()->first();
        $headers = array(
            'Content-Type: application/dwg',
        );
        return Response::download($plan_type->documentDwgPath(), $plan_type->filename_dwg, $headers, 'inline');
    }

    public function delete_plan_type(Request $request)
    {

        $plan_types = Plan_type::whereIn('id', json_decode($request->input('ids')))->get();

        foreach ($plan_types as $plan_type) {
            if ($plan_type->etat == 'En cours') {
                File::deleteDirectory($plan_type->documentDirPath());
                $plan_type->delete();
            }
        }
        return Response::json(['success' => 1], 201);
    }

    public function plan_type_remarques_datatable(Request $request)
    {

        $plan_type_remarques = Plan_type_remarque::where('plan_type_id', $request->plan_type_id);

        return Datatables::of($plan_type_remarques)

            ->addColumn('remarque', function ($plan_type_remarque) {
                return '<input style="vertical-align: middle;position: relative;" class="plant_type_remarque_cb" item_id="' .  $plan_type_remarque->id . '" type="checkbox">&nbsp<a class="plant_type_remarque_a" item_id="' . $plan_type_remarque->id . '" href="javascript:;">' . $plan_type_remarque->remarque . '</a>';
            })
            ->addColumn('date', function ($plan_type_remarque) {
                return Carbon::createFromFormat('Y-m-d H:i:s', $plan_type_remarque->created_at)->format('d/m/Y');
            })

            ->rawColumns(['remarque', 'date'])
            ->make(true);
    }

    public function store_plan_type_remarque(Request $request)
    {
        if (Auth::user()->has_role('Agence urbaine')) {

            $plan_type_remarque = new Plan_type_remarque();
            $plan_type_remarque->id = Str::uuid();
            if ($request->plan_type_remarque_id != '') {
                $plan_type_remarque = Plan_type_remarque::where('id', $request->plan_type_remarque_id)->get();
            } else {
                $plan_type_remarque->user_id = Auth::user()->id;
            }
            $plan_type_remarque->plan_type_id = $request->plan_type_id;
            $plan_type_remarque->remarque = $request->remarque;

            $plan_type_remarque->save();
            return Response::json(['success' => 1], 201);
        } else {
            return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED'], 200);
        }
    }

    public function delete_plan_type_remarques(Request $request)
    {
        $plan_type_remarques = Plan_type_remarque::whereIn('id', json_decode($request->input('ids')))->get();
        foreach ($plan_type_remarques as $plan_type_remarque) {
            if ($plan_type_remarque->user_id == Auth::user()->id) {
                $plan_type_remarque->delete();
            }
        }
        return Response::json(['success' => 1], 201);
    }

    public function validate_plan_type(Request $request)
    {
        if (Auth::user()->has_role('Agence urbaine,Chargé du guichet d\'accompagnement')) {

            $plan_type = Plan_type::where('id', $request->id)->get()->first();
            $plan_type->etat = 'Validé';
            $plan_type->save();
            return Response::json(['success' => 1], 200);
        } else {
            return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED'], 200);
        }
    }
    public function get_plan_type(string $id)
    {
        $plan_type = Plan_type::where('id', $id)->get()->first();

        if (!is_null($plan_type)) {
            return Response::json(['success' => 1, 'data' => $plan_type], 200);
        } else {
            return Response::json(['success' => 0, 'error' => 'NOT_FOUND'], 200);
        }
    }
    public function check_affectation_delection(Request $request)
    {
        if ($request->role == "IGT") {
            $role_column = 'igt_id';
        }
        if ($request->role == "BET") {
            $role_column = 'bet_id';
        }
        if ($request->role == "Architecte") {
            $role_column = 'architecte_id';
        }
        if ($request->role == "Laboratoire") {
            $role_column = 'laboratoire_id';
        }

        $affectation = Affectation::where([['douar_id', $request->douar_id], ['user_id', $request->prestataire_id]])->get()->first();
        if (!is_null($affectation)) {

            if ($request->force_delete == 0) {
                $damandes = Demande::where([[$role_column, $request->prestataire_id], ['douar_id', $request->douar_id]])->get();
                if (count($damandes) > 0) {
                    return Response::json(['success' => 0, 'error' => 'USER_HAS_DEMANDES', 'count' => count($damandes)], 200);
                }
            }

            $affectation->delete();
            return Response::json(['success' => 1], 200);
        } else {
            return Response::json(['success' => 0, 'error' => 'AFFECTATION_NOT_FOUND'], 200);
        }
    }


    public function reaffecter(Request $request)
    {
        if (Auth::user()->has_role('Chef du projet')) {
            $new_prestataire_id = $request->new_prestataire;
            $old_prestataire_id = $request->old_prestataire;

            $new_prestataire = User::where('id', $new_prestataire_id)->get()->first();

            if ($new_prestataire->role->name == "IGT") {
                $role_column = 'igt_id';
            }
            if ($new_prestataire->role->name == "BET") {
                $role_column = 'bet_id';
            }
            if ($new_prestataire->role->name == "Architecte") {
                $role_column = 'architecte_id';
            }
            if ($new_prestataire->role->name == "Laboratoire") {
                $role_column = 'laboratoire_id';
            }
            $douars_ids = json_decode($request->douars);

            $data = [];
            foreach ($douars_ids as $douar_id) {

                $old_affectation = Affectation::where([['douar_id', $douar_id], ['user_id', $old_prestataire_id]])->get()->first();

                $demandes = Demande::where([[$role_column, $old_prestataire_id], ['douar_id', $douar_id]])->update([$role_column => $new_prestataire_id]);

                $dossiers = Dossier::where([[$role_column, $old_prestataire_id], ['douar_id', $douar_id]])->update([$role_column => $new_prestataire_id]);

                $old_affectation->delete();


                $new_affectation = Affectation::where([['douar_id', $douar_id], ['user_id', $new_prestataire_id]])->get()->first();

                if (is_null($new_affectation)) {
                    $new_affectation = new Affectation();
                    $new_affectation->id = Str::uuid();
                    $new_affectation->type = $new_prestataire->role->name;
                    $new_affectation->douar_id = $douar_id;
                    $new_affectation->user_id = $new_prestataire_id;
                    $new_affectation->save();
                }



                $row = new stdClass();
                $row->old_affectation = $old_affectation;
                $row->new_affectation = $new_affectation;
                $row->demandes = $demandes;
                $row->dossiers = $dossiers;



                $data[] = $row;
            }
            return $data;
        } else {
            return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED'], 200);
        }
    }

    public function store_new_douar(Request $request)
    {
        if (Auth::user()->has_role('Administrateur') || Auth::user()->has_role('Chef du projet')) {

            $douar_name = trim($request->douar);

            if (strlen($douar_name) > 0) {
                $douars_check = Douar::where([['nom', $request->douar], ['commune_id', $request->commune]])->count();

                if ($douars_check > 0) {
                    return Response::json(['success' => 0, 'error' => 'DUPLICATED_DOUAR'], 200);
                } else {
                    $douar = new Douar();
                    $douar->id = Str::uuid();
                    $douar->nom = $request->douar;
                    $douar->commune_id = $request->commune;
                    $douar->cercle_id = $request->cercle;
                    $douar->province_id = $request->province;
                    $douar->filiale_id = Province::where('id', $request->province)->get()->first()->filiale_id;
                    $douar->inserted_by_admin = 1;
                    $douar->save();
                    return Response::json(['success' => 1], 200);
                }
            } else {
                return Response::json(['success' => 0, 'error' => 'EMPTY_DOUAR'], 200);
            }
        } else {
            return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED'], 200);
        }
    }

    public function  fix_affectations_demandes_dossier()
    {
        $demandes = Demande::all();
        $dossiers = Dossier::all();

        foreach ($demandes as $demande) {
            $affectations = Affectation::where('douar_id', $demande->douar_id)->get();
            foreach ($affectations as $affectation) {
                if ($affectation->type == "Architecte") {
                    $demande->architecte_id = $affectation->user_id;
                }
                if ($affectation->type == "BET") {
                    $demande->bet_id = $affectation->user_id;
                }
                if ($affectation->type == "IGT") {
                    $demande->igt_id = $affectation->user_id;
                }
                if ($affectation->type == "Laboratoire") {
                    $demande->laboratoire_id = $affectation->user_id;
                }
            }

            $demande->save();
        }

        foreach ($dossiers as $dossier) {
            $affectations = Affectation::where('douar_id', $dossier->douar_id)->get();
            foreach ($affectations as $affectation) {
                if ($affectation->type == "Architecte") {
                    $dossier->architecte_id = $affectation->user_id;
                }
                if ($affectation->type == "BET") {
                    $dossier->bet_id = $affectation->user_id;
                }
                if ($affectation->type == "IGT") {
                    $dossier->igt_id = $affectation->user_id;
                }
                if ($affectation->type == "Laboratoire") {
                    $dossier->laboratoire_id = $affectation->user_id;
                }
            }

            $dossier->save();
        }
        return $demandes;
    }
    public function fix_affectations()
    {
        $douars_architectes = Douar::whereNotNull('architecte_id')->get();
        foreach ($douars_architectes as $douar) {

            $affectation = new Affectation();
            $affectation->id = Str::uuid();
            $affectation->type = 'Architecte';
            $affectation->douar_id = $douar->id;
            $affectation->user_id = $douar->architecte_id;
            $affectation->save();
            $douar->architecte_id = NULL;
            $douar->save();
        }

        $douars_igts = Douar::whereNotNull('igt_id')->get();
        foreach ($douars_igts as $douar) {

            $affectation = new Affectation();
            $affectation->id = Str::uuid();
            $affectation->type = 'IGT';
            $affectation->douar_id = $douar->id;
            $affectation->user_id = $douar->igt_id;
            $affectation->save();
            $douar->igt_id = NULL;
            $douar->save();
        }

        $douars_bets = Douar::whereNotNull('bet_id')->get();
        foreach ($douars_bets as $douar) {

            $affectation = new Affectation();
            $affectation->id = Str::uuid();
            $affectation->type = 'BET';
            $affectation->douar_id = $douar->id;
            $affectation->user_id = $douar->bet_id;
            $affectation->save();
            $douar->bet_id = NULL;
            $douar->save();
        }

        $douars_lobos = Douar::whereNotNull('laboratoire_id')->get();
        foreach ($douars_lobos as $douar) {

            $affectation = new Affectation();
            $affectation->id = Str::uuid();
            $affectation->type = 'Laboratoire';
            $affectation->douar_id = $douar->id;
            $affectation->user_id = $douar->laboratoire_id;
            $affectation->save();
            $douar->laboratoire_id = NULL;
            $douar->save();
        }

        $douars_bcs = Douar::whereNotNull('bc_id')->get();
        foreach ($douars_bcs as $douar) {

            $affectation = new Affectation();
            $affectation->id = Str::uuid();
            $affectation->type = 'BC';
            $affectation->douar_id = $douar->id;
            $affectation->user_id = $douar->bc_id;
            $affectation->save();
            $douar->bc_id = NULL;
            $douar->save();
        }

        $affectations = Affectation::all();

        foreach ($affectations as $affectation) {

            if (User::where('id', $affectation->user_id)->count() == 0) {
                $affectation->delete();
            }
        }
    }

    public function clean_affectations()
    {
        $data = [];
        $affectations = Affectation::where('type', 'Architecte')->get();
        foreach ($affectations as  $affectation) {
            $count = Demande::where([['architecte_id', $affectation->user_id]])->count();

            $user = User::where('id', $affectation->user_id)->get()->first();
            $douar = Douar::where('id', $affectation->douar_id)->get()->first();

            if ($count == 0) {
                $user_data = new stdClass();
                $user_data->user =  $user->organisme_ste;
                $user_data->douar = $douar->nom_ar;
                $user_data->count = $count;
                $data[] = $user_data;
            }
        }

        return $data;
    }

    public function clean_affectations_delete()
    {

        $affectations = Affectation::where('type', 'Architecte')->get();

        foreach ($affectations as  $affectation) {
            $count = Affectation::where([['douar_id', $affectation->douar_id], ['user_id', $affectation->user_id]])->count();
            if ($count > 1) {
                $affectation->delete();
            }
        }
        return 1;
    }


    public function fix_documents_with_no_dossiers()
    {
        $now = Carbon::now();

        $data = [];
        $documents = Dossier_document::all();
        foreach ($documents as  $document) {
            $count = Dossier::where([['id', $document->dossier_id]])->count();
            if ($count == 0) {
                $row = new stdClass();
                $row->dossier_id = $document->dossier_id;
                $row->created_at = Carbon::createFromFormat('Y-m-d H:i:s',  $document->created_at)->format('d/m/Y H:i:s');
                $doc_date = Carbon::parse($document->created_at);
                $diff = $doc_date->diffInHours($now);
                $row->diff_time = $diff;

                if ($diff >= 24) {
                    $document->delete();
                }
                $data[] = $row;
            }
        }
        return $data;
    }


    public function carte_aleas_datatable(Request $request)
    {
        $data = [];
        $files = scandir(base_path('files/aleas/' . $request->path));
        unset($files[0], $files[1]);

        foreach ($files as $file) {
            $row = new stdClass();
            $full_path = base_path('files/aleas/' . $request->path . '/' . $file);
            $path_info = pathinfo($full_path);
            if (is_dir($full_path)) {
                $row->type = "DIR";
                $row->path =   $request->path . '/' . $file;
                $row->name = $path_info['filename'];
                $data[] = $row;
            }
        }

        foreach ($files as $file) {
            $row = new stdClass();
            $full_path = base_path('files/aleas/' . $request->path . '/' . $file);
            $path_info = pathinfo($full_path);
            if (is_file($full_path)) {
                $row->type = "FILE";
                $row->extension = $path_info['extension'];
                $row->size = filesize($full_path);
                $row->path =   $request->path . '/' . $file;
                $row->name =   $file;
                $data[] = $row;
            }
        }

        return Datatables::of($data)
            ->editColumn('data', function ($file) {

                if ($file->type == "DIR") {
                    $html = '<a class="carte_aleas_dir" path="' . $file->path . '" href="javascript:;"><i style="color: coral; font-size: 1.25rem; margin: 8px;" class="fa-solid fa-folder"></i>&nbsp<span>' . $file->name . "</span></a>";
                    return $html;
                } else {

                    $icon = 'fa-solid fa-file';
                    $color = "#1492C8";

                    if ($file->extension == 'docx') {
                        $icon = 'fa-solid fa-file-word';
                        $color = "#1C76FD";
                    }
                    if ($file->extension == 'xls') {
                        $icon = 'fa-solid fa-file-excel';
                        $color = "#32BF0F";
                    }
                    if ($file->extension == 'xlsx') {
                        $icon = 'fa-solid fa-file-excel';
                        $color = "#32BF0F";
                    }

                    if ($file->extension == 'pdf') {
                        $icon = 'fa-solid fa-file-pdf';
                        $color = "#FF3136";
                    }

                    if (
                        $file->extension == 'jpeg' ||
                        $file->extension == 'png' ||
                        $file->extension == 'tiff' ||
                        $file->extension == 'bmp' ||
                        $file->extension == 'jpg'

                    ) {
                        $icon = 'fa-solid fa-file-image';
                        $color = "#8A51D0";
                    }
                    $html = '<div class="row">';

                    $html .= '<div style="padding-left: 0px;" class="col-6">';
                    $html .= '<i style="color: ' . $color . '; font-size: 1.25rem; margin: 8px;" class="' . $icon . '"></i>&nbsp<span>' . $file->name . "</span>";
                    $html .= '</div>';

                    $html .= '<div class="col-3">';
                    $html .= $this->formatBytes($file->size);
                    $html .= '</div>';
                    $html .= '<div class="col-3">';
                    $html .= '<a style="color: #33a433 !important;" class="carte_aleas_download_file" path="' . $file->path . '" href="javascript:;">Télécharger</a>';
                    $html .= '</div>';
                    $html .= '</div>';
                    return $html;
                }
            })

            ->rawColumns(['data'])
            ->make(true);
    }

    public function  carte_aleas_download(Request $request)
    {

        $file_path = base_path('files/aleas/' . $request->path);
        $path_info = pathinfo($file_path);

        return Response::download($file_path, $path_info['basename']);
    }

    public static function formatBytes($size, $precision = 2)
    {
        if ($size > 0) {
            $size = (int) $size;
            $base = log($size) / log(1024);
            $suffixes = array(' bytes', ' KB', ' MB', ' GB', ' TB');

            return round(pow(1024, $base - floor($base)), $precision) . $suffixes[floor($base)];
        } else {
            return $size;
        }
    }

    public static function export_douars_data()
    {
        $data = [];
        $douars =  Douar::all();

        foreach ($douars as $douar) {
            $row = new stdClass();

            $row->douar_ar = $douar->nom_ar;
            $row->commune_ar = $douar->commune->nom;
            $row->cercle = $douar->commune->cercle->nom;
            $row->province = $douar->commune->cercle->province->nom;

            $row->douar_id = $douar->id;
            $row->commune_id = $douar->commune->id;
            $row->cercle_id = $douar->commune->cercle->id;
            $row->province_id =  $douar->commune->cercle->province->id;

            $data[] = $row;
        }

        return  $data;
    }

    public static function test_subdir_max()
    {
        for ($i = 1; $i <= 5000000; $i++) {
            $dir_path = base_path('test_dir/' . $i);
            mkdir($dir_path);
        }
    }

    public static function dump_files()
    {
        $dossier_docs = Dossier_document::where('sync', 0)->get();
        foreach ($dossier_docs as  $dossier_doc) {
            if (!is_null($dossier_doc->dossier_id)) {

                $local_path = base_path('files/dossiers_docs/' . $dossier_doc->id . '/' . $dossier_doc->filename);
                $exists_local = File::exists($local_path);
                if ($exists_local) {

                    try {
                        Storage::disk('sftp')->put('batir/dossiers_docs/' . $dossier_doc->id . '/' . $dossier_doc->id, file_get_contents($local_path));

                        $dossier_doc->sync = 1;
                        $dossier_doc->save();
                    } catch (Exception $e) {
                        throw new Exception($e->getMessage());
                    }
                }
            }
        }
    }

    public static function dump_files_rehab()
    {
        $dossier_docs = DossierRehab_document::where('sync', 0)->get();
        foreach ($dossier_docs as  $dossier_doc) {
            if (!is_null($dossier_doc->dossier_id)) {

                $local_path = base_path('files/dossiers_rehab_docs/' . $dossier_doc->id . '/' . $dossier_doc->filename);
                $exists_local = File::exists($local_path);
                if ($exists_local) {

                    try {
                        Storage::disk('sftp')->put('batir/dossiers_rehab_docs/' . $dossier_doc->id . '/' . $dossier_doc->id, file_get_contents($local_path));

                        $dossier_doc->sync = 1;
                        $dossier_doc->save();
                    } catch (Exception $e) {
                        throw new Exception($e->getMessage());
                    }
                }
            }
        }
    }

    public static function dump_files_demandes()
    {
        $demande_docs = Demande_document::where('sync', 0)->get();
        foreach ($demande_docs as  $demande_doc) {
            if (!is_null($demande_doc->demande_id)) {

                $local_path = base_path('files/demandes_docs/' . $demande_doc->id . '/' . $demande_doc->filename);
                $exists_local = File::exists($local_path);
                if ($exists_local) {

                    try {
                        Storage::disk('sftp')->put('batir/demandes_docs/' . $demande_doc->id . '/' . $demande_doc->id, file_get_contents($local_path));

                        $demande_doc->sync = 1;
                        $demande_doc->save();
                    } catch (Exception $e) {
                        throw new Exception($e->getMessage());
                    }
                }
            }
        }
    }

    public static function dump_files_r2()
    {
        $dossier_docs = Dossier_document::where('sync_r2', 0)->get();

        foreach ($dossier_docs as  $dossier_doc) {
            if (!is_null($dossier_doc->dossier_id)) {
                //$dossier_count = Dossier::where('id', $dossier_doc->dossier_id)->count();
                // if ($dossier_count > 0) {

                // }

                $local_path = base_path('files/dossiers_docs/' . $dossier_doc->id . '/' . $dossier_doc->filename);
                $exists_local = File::exists($local_path);
                if ($exists_local) {

                    try {
                        Storage::disk('r2')->put('dossiers_docs/' . $dossier_doc->id . '/' . $dossier_doc->id, file_get_contents($local_path));

                        $dossier_doc->sync_r2 = 1;
                        $dossier_doc->save();
                    } catch (Exception $e) {
                        throw new Exception($e->getMessage());
                    }
                }
            }
        }
    }

    public static function dump_files_rehab_r2()
    {
        $dossier_docs = DossierRehab_document::where('sync_r2', 0)->get();

        foreach ($dossier_docs as  $dossier_doc) {
            if (!is_null($dossier_doc->dossier_id)) {
                $local_path = base_path('files/dossiers_rehab_docs/' . $dossier_doc->id . '/' . $dossier_doc->filename);
                $exists_local = File::exists($local_path);
                if ($exists_local) {
                    try {
                        Storage::disk('r2')->put('dossiers_rehab_docs/' . $dossier_doc->id . '/' . $dossier_doc->id, file_get_contents($local_path));
                        $dossier_doc->sync_r2 = 1;
                        $dossier_doc->save();
                    } catch (Exception $e) {
                        throw new Exception($e->getMessage());
                    }
                }
            }
        }
    }

    public static function dump_files_demandes_r2()
    {
        $demande_docs = Demande_document::where('sync_r2', 0)->get();

        foreach ($demande_docs as  $demande_doc) {
            if (!is_null($demande_doc->demande_id)) {
                $local_path = base_path('files/demandes_docs/' . $demande_doc->id . '/' . $demande_doc->filename);
                $exists_local = File::exists($local_path);
                if ($exists_local) {
                    try {
                        Storage::disk('r2')->put('demandes_docs/' . $demande_doc->id . '/' . $demande_doc->id, file_get_contents($local_path));
                        $demande_doc->sync_r2 = 1;
                        $demande_doc->save();
                    } catch (Exception $e) {
                        throw new Exception($e->getMessage());
                    }
                }
            }
        }
    }

    public static function check_files_r2()
    {
        $dossier_docs = Dossier_document::where([['sync_r2', 1], ['sync_r2_check', 0]])->get();
        foreach ($dossier_docs as  $dossier_doc) {
            $exists_r2 = Storage::disk('r2')->exists('dossiers_docs/' . $dossier_doc->id . '/' . $dossier_doc->id);

            if ($exists_r2) {
                $dossier_doc->sync_r2_check = 1;
                $dossier_doc->save();
            } else {
                $dossier_doc->sync_r2_check = 0;
                $dossier_doc->save();
            }
        }
    }


    public static function get_mi()
    {

        $file = Storage::disk('sftp_mi')->get('Seisme_Liste bénéciciaires des avances Const_Rehab_25042024_1326.xlsx');
        $response = Response::make($file, 200);
        $response->header("Content-Type", 'application/xlsx');
        return $response;
    }

    public static function alomran_get_dossier_document(string $id)
    {
        $dossier_doc = Dossier_document::where('id', $id)->get()->first();

        if (is_null($dossier_doc)) {
            return Response::json(['success' => 0, 'error' => 'NOT_FOUND'], 200);
        } else {
            $file = Storage::disk('sftp')->get('batir/dossiers_docs/' . $dossier_doc->id . '/' . $dossier_doc->filename);
            $response = Response::make($file, 200);
            if ($dossier_doc->file_type == 'PDF') {
                $response->header("Content-Type", 'application/pdf');
            }
            if ($dossier_doc->file_type == 'JPG') {
                $response->header("Content-Type", 'image/jpg');
            }
            if ($dossier_doc->file_type == 'PNG') {
                $response->header("Content-Type", 'image/png');
            }
            if ($dossier_doc->file_type == 'JPEG') {
                $response->header("Content-Type", 'image/jpeg');
            }
            if ($dossier_doc->file_type == 'ZIP') {
                $response->header("Content-Type", 'application/zip');
            }
            if ($dossier_doc->file_type == 'RAR') {
                $response->header("Content-Type", 'application/rar');
            }
            if ($dossier_doc->file_type == 'GEOJSON') {
                $response->header("Content-Type", 'application/geojson');
            }
            if ($dossier_doc->file_type == 'DWG') {
                $response->header("Content-Type", 'application/dwg');
            }
            return $response;
        }
    }

    public static function clean_files()
    {


        // $result = new stdClass();
        // $result->count = 0;
        // $result->total_size = 0;

        // $dossier_docs = Dossier_document::all();
        // foreach ($dossier_docs as  $dossier_doc) {

        //     if (is_null($dossier_doc->dossier_id)) {
        //         $result->count++;
        //         $result->total_size += $dossier_doc->file_size;
        //     } else {
        //         $dossier_count = Dossier::where('id', $dossier_doc->dossier_id)->count();
        //         if ($dossier_count == 0) {
        //             $result->count++;
        //             $result->total_size += $dossier_doc->file_size;
        //         }
        //     }
        // }

        // return $result;
    }

    public static function isValidCINFormat($inputString)
    {
        $pattern = '/^[a-zA-Z]{1,2}\d{2,}$/';
        if (preg_match($pattern, $inputString)) {

            return true;
        } else {
            return false;
        }
    }

    public static function detect_bad_cin()
    {
        $data = [];
        $Petitionnaires =  Petitionnaire::all();

        foreach ($Petitionnaires as $Petitionnaire) {
            $valid_cin = self::isValidCINFormat($Petitionnaire->cin);

            if (!$valid_cin) {
                $row = new stdClass();
                $row->cin = $Petitionnaire->cin;
                $row->nom = $Petitionnaire->nom;
                $row->prenom = $Petitionnaire->prenom;
                $row->tel = $Petitionnaire->tel;
                $data[] =  $row;
            }
        }

        return  $data;
    }

    public static function optimise_petitionnaires()
    {
        $data = [];
        $Petitionnaires =  Petitionnaire::all();

        foreach ($Petitionnaires as $Petitionnaire) {
            $nom_is_ar = self::isArabic($Petitionnaire->nom);
            $prenom_is_ar = self::isArabic($Petitionnaire->prenom);

            $row = new stdClass();
            $row->nom = $Petitionnaire->nom;
            $row->nom_is_ar =  $nom_is_ar;

            $row->prenom = $Petitionnaire->prenom;
            $row->prenom_is_ar =  $prenom_is_ar;
            $data[] = $row;
        }

        return  $data;
    }

    public static function isArabic($string)
    {
        if (preg_match('/\p{Arabic}/u', $string))
            return true;
        return false;
    }

    public static function detect_bad_demandes()
    {
        $data = [];
        $demandes =  Demande::with('petitionnaires')->get();

        foreach ($demandes as $demande) {
            $petitionnaires_count =  count($demande->petitionnaires);
            if ($petitionnaires_count == 0) {
                $data[] = $demande;
                $demande->delete();
            }
        }

        return  $data;
    }

    public static function detect_cin_no_demandes()
    {
        $data = [];
        $petitionnaires =  Petitionnaire::all();

        foreach ($petitionnaires as $petitionnaire) {
            $demande = Demande::where('id', $petitionnaire->demande_id)->get()->first();
            if (is_null($demande)) {
                $data[] = $petitionnaire;
                $petitionnaire->delete();
            }
        }

        return  $data;
    }

    public static function dump_etude_geotechnique_geometry()
    {
        $etudes_geotechniques_conclusions = Etude_geotechnique_conclusion::all();

        $result = [];


        foreach ($etudes_geotechniques_conclusions as $etude_geotechnique_conclusion) {

            $douar = Douar::where('etude_geotechnique_id', $etude_geotechnique_conclusion->etude_geotechnique_id)->get()->first();

            // if (!is_null($douar)) {
            //     $douar_path = base_path('files/etudes_geotechniques/' . $douar->nom);
            //     if (!file_exists($douar_path)) {
            //         mkdir($douar_path);
            //     }
            //     $decision_path = base_path('files/etudes_geotechniques/' . $douar->nom . '/' . $etude_geotechnique_conclusion->conclusion);
            //     if (!file_exists($decision_path)) {
            //         mkdir($decision_path);
            //     }

            //     $file_path = base_path('files/etudes_geotechniques/' . $douar->nom . '/' . $etude_geotechnique_conclusion->conclusion . '/' . $etude_geotechnique_conclusion->id . '.geojson');
            //     $file = fopen($file_path, "w");
            //     fwrite($file, json_encode($constructible[] = $etude_geotechnique_conclusion->zone));
            //     fclose($file);
            // }
            if (!is_null($douar)) {
                // if ($etude_geotechnique_conclusion->conclusion == 'Constructible') {
                //     $constructible[] = $etude_geotechnique_conclusion->zone;
                // }
                // if ($etude_geotechnique_conclusion->conclusion == 'Constructible à conditions') {
                //     $constructible_a_conditions[] = $etude_geotechnique_conclusion->zone;
                // }
                // if ($etude_geotechnique_conclusion->conclusion == 'Constructible à conditions strictes') {
                //     $constructible_a_conditions_strictes[] = $etude_geotechnique_conclusion->zone;
                // }
                // if ($etude_geotechnique_conclusion->conclusion == 'Non constructible') {
                //     $non_constructible[] = $etude_geotechnique_conclusion->zone;
                // }
                $zone_polygon = \geoPHP::load($etude_geotechnique_conclusion->zone, 'geojson');

                $row = new stdClass();

                $row->douar = $douar->nom;
                $row->zone = $etude_geotechnique_conclusion->conclusion;
                $row->wkt = $zone_polygon->out('wkt');

                $result[] = $row;
            }
        }

        // $result->constructible = $constructible;
        // $result->constructible_a_conditions = $constructible_a_conditions;
        // $result->constructible_a_conditions_strictes = $constructible_a_conditions_strictes;
        // $result->non_constructible = $non_constructible;

        return $result;
    }



    public static function update_petitionnaires()
    {

        LPetitionnaire::truncate();
        LPetitionnaire_dr::truncate();

        $Petitionnaires =  Petitionnaire::all()->toArray();;

        foreach ($Petitionnaires as  $Petitionnaire) {
            LPetitionnaire::insert($Petitionnaire);
        }

        $Petitionnaires_dr =  Petitionnaire_dr::all()->toArray();;

        foreach ($Petitionnaires_dr as  $Petitionnaire_dr) {
            LPetitionnaire_dr::insert($Petitionnaire_dr);
        }
        return 1;
    }

    public static function cleanFiles()
    {
        $dossiers_documents = Dossier_document::where('active', 0)->whereNull('dossier_id')->get();
        $demandes_documents = Demande_document::whereNull('demande_id')->get();
        $dossiers_rehab_documents = DossierRehab_document::where('active', 0)->whereNull('dossier_id')->get();


        $result = new stdClass();
        $result->dossiers_documents_total_size = 0;
        $result->demandes_documents_total_size = 0;
        $result->dossiers_rehab_documents_total_size = 0;
        $result->total_size = 0;

        foreach ($dossiers_documents as  $document) {

            $result->dossiers_documents_total_size += $document->file_size;
        }
        foreach ($demandes_documents as  $document) {

            $result->demandes_documents_total_size += $document->file_size;
        }
        foreach ($dossiers_rehab_documents as  $document) {

            $result->dossiers_rehab_documents_total_size += $document->file_size;
        }
        $result->total_size =  $result->dossiers_documents_total_size + $result->demandes_documents_total_size + $result->dossiers_rehab_documents_total_size;
        return  $result;
    }
}
