<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Auth;
use App\Models\Geoportail;
use Illuminate\Http\Request;
use App\Models\CompteIGT;
use App\Models\User;
use App\Models\Douar;
use App\Models\Declaration;
use App\Models\Liquidation;
use App\Models\Proprietaire;
use App\Models\Terrain_document;
use App\Models\Terrain;
use App\Models\Zone;
use App\Models\ZonePA;
use App\Models\TerrainOperation;
use Carbon\Carbon;
use DataTables;
use stdClass;

class GeoportailController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $zones = Zone::where('annee', date('Y'))->get();
        return view('geoportail.index', [
            'zones' => $zones
        ]);
    }

    public function get_etude_geo_geometries(Request $request)
    {
        $douars_id = array_unique(explode(',', $request->douars));
        $result = [];
        $douars = Douar::whereIn('id', $douars_id)->with('etude_geotechnique.etudes_geotechniques_conclusions')->get();

        foreach ($douars as  $douar) {

            if (!is_null($douar->etude_geotechnique)) {
                if (count($douar->etude_geotechnique->etudes_geotechniques_conclusions) > 0) {
                    $element = new stdClass();
                    $element->douar_id = $douar->id;
                    $element->nom = $douar->nom;
                    $element->geometry = $douar->etude_geotechnique->etudes_geotechniques_conclusions;
                    $result[] = $element;
                }
            }
        }
        return $result;
    }

    public function map_get_terrains(Request $request)
    {
        $result = [];

        $filters = json_decode($request->filters);
        $terrains = Terrain::orderBy('created_at', 'desc')->get();

        foreach ($filters as $filter) {
            if ($filter->column == 'zone') {
                $terrains = $terrains->whereHas('zones', function ($query) use ($filter) {
                    $query->where('zones.id', $filter->value);
                });
                continue;
            }
            if ($filter->column === 'declaration') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }

            if ($filter->column === 'liquidation') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }
            if ($filter->column == 'terrains.created_at') {

                $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
                $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));
                $to->addDay();
                $terrains = $terrains->whereBetween($filter->column, [$from, $to]);
            } else {
                if ($filter->table == '') {
                    $terrains = $terrains->where($filter->column, '=', $filter->value);
                } else {
                    $terrains = $terrains->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, '=', $filter->value);
                    });
                }
            }
        }




        foreach ($terrains as $terrain) {
            $plle = new stdClass();
            $plle->id = $terrain->id;
            $plle->article = $terrain->article;
            $plle->geojson = $terrain->geojson;
            $result[] = $plle;
        }


        return $result;
    }



    public function getAllTerrains(Request $request)
    {
        $filters = json_decode($request->filters);

        $terrains = Terrain::with([
            'zones' => function ($query) {
                $query->where('annee', date('Y'));
            },
            'zones_pa'
        ]);

        foreach ($filters as $filter) {

            if ($filter->column == 'zone') {
                $terrains = $terrains->whereHas('zones', function ($query) use ($filter) {
                    $query->where('zones.id', $filter->value);
                });
                continue;
            }
            if ($filter->column === 'declaration') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }

            if ($filter->column === 'liquidation') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }
            if ($filter->column == 'terrains.created_at') {
            } else {
                if ($filter->table == '') {
                    $terrains = $terrains->where($filter->column, '=', $filter->value);
                } else {
                    $terrains = $terrains->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, '=', $filter->value);
                    });
                }
            }
        }

        $terrains = $terrains->get();


        foreach ($terrains as $terrain) {
            $has_declaration = Declaration::where('id_terrain', $terrain->id)
                ->whereYear('date', date('Y'))
                ->exists();

            if ($has_declaration) {
                $terrain->declation_html = '<span style="color: #558B2F">Déclaré</span>';
            } else {
                $terrain->declation_html =  '<span style="color: #EF5350">Non déclaré</span>';
            }


            $has_liquidation = Liquidation::where('id_terrain', $terrain->id)
                ->whereYear('date', date('Y'))
                ->exists();

            if ($has_liquidation) {
                $terrain->liquidation_html =  '<span style="color: #558B2F">Liquidé</span>';
            } else {
                 $terrain->liquidation_html =  '<span style="color: #EF5350">Non liquidé</span>';
            }
        }
        return response()->json([
            'success' => true,
            'terrains' => $terrains
        ]);
    }

    public function terrains_datatable(Request $request)
    {
        $filters = json_decode($request->filters);

        $terrains = Terrain::with('zones');

        foreach ($filters as $filter) {

            if ($filter->column == 'zone') {
                $terrains = $terrains->whereHas('zones', function ($query) use ($filter) {
                    $query->where('zones.id', $filter->value);
                });
                continue;
            }
            if ($filter->column === 'declaration') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }

            if ($filter->column === 'liquidation') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }
            if ($filter->column == 'terrains.created_at') {
            } else {
                if ($filter->table == '') {
                    $terrains = $terrains->where($filter->column, '=', $filter->value);
                } else {
                    $terrains = $terrains->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, '=', $filter->value);
                    });
                }
            }
        }

        $terrains = $terrains->limit($request->limit)->offset($request->offset);


        return Datatables::of($terrains)
            ->addColumn('row', function ($terrain) {
                $accent_color = $terrain->type_terrain_color($terrain->type);
                // Get unique zones names for this terrain (latest year for each zone)
                // Calculate centroid from GeoJSON
                $centroid = $this->calculateCentroid($terrain->geojson);
                $uniqueZones = $terrain->zones
                    ->groupBy('nom')
                    ->map(function ($zones) {
                        return $zones->sortByDesc('annee')->first();
                    });
                $zonesNames = $uniqueZones->pluck('nom')->join(', ');

                // Create zones display
                $zonesHtml = '';
                if ($uniqueZones->count() > 0) {
                    $zonesHtml = '<div style="margin-top: 5px; padding-top: 5px; border-top: 1px solid #e0e0e0;">';
                    $zonesHtml .= '<small style="color: #666; font-weight: bold;">Zones:</small><br>';

                    foreach ($uniqueZones as $zone) {
                        $zonesHtml .= '<span style="display: inline-block; background: #f0f0f0; padding: 2px 6px; margin: 2px; border-radius: 3px; font-size: 11px; color: #555;">';
                        $zonesHtml .= $zone->nom . ' (' . number_format($zone->tarif, 2) . ' DH - ' . $zone->annee . ')';
                        $zonesHtml .= '</span>';
                    }
                    $zonesHtml .= '</div>';
                }

                $html = '<div  class="row dossier_row" lat="' . $centroid['lat'] . '" lng="' . $centroid['lng'] . '" item_id="' . $terrain->id . '"  >
                            <div class="col-2" style="padding:0px;opacity:0.8; background: ' . $accent_color . ';border-radius: 6px 0px 0px 6px;border-right: 1px #9d9d9d dashed;">
                                <i style="color: ' . $accent_color . ' !important" class="fa-regular fa-location-dot dossier_row_icon"></i>
                            </div>
                            <div class="col-10">
                            <b style="font-family: \'Roboto\'; color: #49669a; border-bottom: 1px #b7b7b7 dashed; width: 100%; display: block;"> ' . $terrain->ref  . ' <b style="font-family: \'Roboto\'; color: #787878; float: right; display: block; font-weight: 400;"> ' . $zonesNames . '</b></b>

                           
                            <span style="color: ' . $accent_color . '; font-weight: 400; font-family: Roboto;">' . $terrain->type . '</span>
                            </div>     
                        </div>';

                return $html;
            })


            ->rawColumns(['row'])
            ->only(['row'])
            ->make(true);
    }


    private function calculateCentroid($geojson)
    {
        // Default values in case of error
        $defaultCentroid = ['lat' => 0, 'lng' => 0];

        try {
            // Parse GeoJSON
            $geometry = json_decode($geojson, true);

            if (!$geometry || !isset($geometry['coordinates']) || !is_array($geometry['coordinates'])) {
                return $defaultCentroid;
            }

            // Handle different geometry types
            $coordinates = null;

            if ($geometry['type'] === 'Polygon') {
                // For Polygon, take the first ring (exterior ring)
                $coordinates = $geometry['coordinates'][0];
            } elseif ($geometry['type'] === 'MultiPolygon') {
                // For MultiPolygon, take the first polygon's first ring
                $coordinates = $geometry['coordinates'][0][0];
            } else {
                return $defaultCentroid;
            }

            if (!$coordinates || count($coordinates) < 3) {
                return $defaultCentroid;
            }

            // Calculate centroid using the shoelace formula
            $latSum = 0;
            $lngSum = 0;
            $area = 0;

            $len = count($coordinates) - 1; // Exclude the last point (same as first)

            for ($i = 0; $i < $len; $i++) {
                $x1 = $coordinates[$i][0]; // longitude
                $y1 = $coordinates[$i][1]; // latitude
                $x2 = $coordinates[($i + 1) % $len][0]; // longitude
                $y2 = $coordinates[($i + 1) % $len][1]; // latitude

                $factor = ($x1 * $y2 - $x2 * $y1);
                $lngSum += ($x1 + $x2) * $factor;
                $latSum += ($y1 + $y2) * $factor;
                $area += $factor;
            }

            $area = $area / 2;

            // Avoid division by zero
            if ($area == 0) {
                // If area is 0, calculate simple average
                $avgLat = array_sum(array_column($coordinates, 1)) / count($coordinates);
                $avgLng = array_sum(array_column($coordinates, 0)) / count($coordinates);
                return ['lat' => $avgLat, 'lng' => $avgLng];
            }

            $centroidLng = $lngSum / (6 * $area);
            $centroidLat = $latSum / (6 * $area);

            return ['lat' => $centroidLat, 'lng' => $centroidLng];
        } catch (Exception $e) {
            // Log error if needed
            \Log::error('Error calculating centroid: ' . $e->getMessage());
            return $defaultCentroid;
        }
    }
}
