<?php

namespace App\Http\Controllers;

use App\Models\Adresse;
use App\Models\Affectation;
use App\Models\Cercle;
use App\Models\Commune;
use Illuminate\Support\Facades\DB;
use App\Models\Terrain;
use App\Models\Dossier;
use App\Models\User;
use App\Models\Declaration;
use App\Models\Liquidation;
use App\Models\Proprietaire;
use App\Models\Terrain_document;
use App\Models\Zone;
use App\Models\ZonePA;
use App\Models\TerrainOperation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Str;
use DataTables;
use Illuminate\Support\Facades\File;
use Response;
use \stdClass;
use PhpOffice\PhpSpreadsheet\IOFactory;

include(app_path() . '/Http/Controllers/geoPHP.inc');
class TerrainsController extends Controller
{



    public function index()
    {
        $zones = Zone::where('annee', date('Y'))->get();
        return view('terrains.index', [
            'zones' => $zones
        ]);
    }


    public function index_terrains_taxables()
    {
        $zones = Zone::where('annee', date('Y'))->get();
        return view('terrains.index_terrains_taxables', [
            'zones' => $zones
        ]);
    }

    public function index_exonerations_temporaires()
    {
        $zones = Zone::where('annee', date('Y'))->get();
        return view('terrains.index_exonerations_temporaires', [
            'zones' => $zones
        ]);
    }


    public function index_supprimes()
    {
        $zones = Zone::where('annee', date('Y'))->get();
        return view('terrains.index_terrains_supprimes', [
            'zones' => $zones
        ]);
    }



    public function getAllTerrains(Request $request)
    {
        $terrains = Terrain::with([
            'zones' => function ($query) {
                $query->where('annee', date('Y'));
            },
            'zones_pa'
        ])->get();

        return response()->json([
            'success' => true,
            'terrains' => $terrains
        ]);
    }





    public function terrains_datatable(Request $request)
    {
        $filters = json_decode($request->filters);
        $terrains = Terrain::with(['zones', 'proprietaires', 'zones_pa'])->orderBy('created_at', 'desc');

        foreach ($filters as $filter) {
            if (isset($filter->params)) {
                $params = explode(',', $filter->params);
                foreach ($params as $param) {
                    if ($param == "remove_spaces") {
                        $filter->value = str_replace(' ', '', $filter->value);
                    }
                }
            }

            if ($filter->column === 'search') {
                $searchValue = $filter->value;

                $terrains = $terrains->where(function ($query) use ($searchValue) {
                    $query->where('ref', 'LIKE', '%' . $searchValue . '%')
                        ->orWhereHas('proprietaires', function ($subQuery) use ($searchValue) {
                            $subQuery->where('type', 'morale')
                                ->where('rc', 'LIKE', '%' . $searchValue . '%');
                        })
                        ->orWhereHas('proprietaires', function ($subQuery) use ($searchValue) {
                            $subQuery->where('type', 'physique')
                                ->where('cine', 'LIKE', '%' . $searchValue . '%');
                        });
                });
                continue;
            }

            if ($filter->column == 'type') {
                if ($filter->value == "NULL") {
                    $terrains =  $terrains->whereNull("type");
                } else {
                    $terrains =  $terrains->where("type", $filter->value);
                }
                continue;
            }

            if ($filter->column == 'zone') {
                $terrains = $terrains->whereHas('zones', function ($query) use ($filter) {
                    $query->where('zones.id', $filter->value);
                });
                continue;
            }

            if ($filter->column == 'proprietaire_type') {
                $terrains = $terrains->whereHas('proprietaires', function ($query) use ($filter) {
                    $query->where('type', $filter->value);
                });
                continue;
            }

            if ($filter->column == 'qualite_redevable') {
                $terrains = $terrains->whereHas('proprietaires.qualites', function ($query) use ($filter) {
                    $query->where('qualite', $filter->value);
                });
                continue;
            }


            if ($filter->column === 'declaration') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }

            if ($filter->column === 'liquidation') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }

            if ($filter->column == 'created_at') {

                $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
                $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));

                $terrains = $terrains->whereBetween($filter->column, [$from, $to]);
            } else {
                if ($filter->table == '') {
                    $terrains = $terrains->where($filter->column, '=', $filter->value);
                } else {
                    $terrains = $terrains->whereHas($filter->table, function ($query) use ($filter) {
                        $query->where($filter->table . '.' . $filter->column, '=', $filter->value);
                    });
                }
            }
        }

        return Datatables::eloquent($terrains)
            ->editColumn('article', function ($terrain) {
                return '<i style="font-size: 0.8rem;color: #7C7C7C;" class="fa-regular fa-map-marker-alt"></i>&nbsp<a  href="' . asset('') . 'terrains/show/' . $terrain->id . '">' . $terrain->article . '</a>&nbsp';
            })
            ->addColumn('ref', function ($terrain) {
                return $terrain->ref;
            })
            ->addColumn('surface', function ($terrain) {
                return number_format($terrain->surface, 0, '.', ' ') . 'm²';
            })
            ->addColumn('statut_foncier', function ($terrain) {
                return $terrain->statut_foncier;
            })

            ->addColumn('adresse', function ($terrain) {
                return $terrain->adresse;
            })


            ->addColumn('zones', function ($terrain) {
                $currentYear = date('Y');
                $zones = $terrain->zones->where('annee', $currentYear);

                if ($zones->count() > 0) {
                    return $zones->pluck('nom')->implode(', ');
                } else {
                    return '<b style="color:gray">Néant</b>';
                }
            })
            // ->addColumn('zones_pa', function ($terrain) {
            //     $zones_pa = $terrain->zones_pa;

            //     if ($zones_pa->count() > 0) {
            //         return $zones_pa->pluck('nom')->implode(', ');
            //     } else {
            //         return '<b style="color:gray">Néant</b>';
            //     }
            // })
            ->addColumn('proprietaires', function ($terrain) {
                $proprietaires = $terrain->proprietaires;
                $html = '<b style="color:gray">Néant</b>';
                if ($proprietaires->count() > 0) {
                    $html = '<select class="form-control select2nc">';
                    foreach ($proprietaires as $prop) {
                        $typeInfo = '';
                        if ($prop->type === 'physique') {
                            $typeInfo = ' | CINE: ' . ($prop->cine ?? 'N/A');
                        } elseif ($prop->type === 'morale') {
                            $typeInfo = ' | RC: ' . ($prop->rc ?? 'N/A') . ' | Ville: ' . ($prop->ville_rc ?? 'N/A');
                        }
                        $html .= '<option>' . $prop->nom . ' ' . $prop->prenom . $typeInfo . '</option>';
                    }
                    $html .= '</select>';
                }
                return $html;
            })
            ->addColumn('type', function ($terrain) {
                switch ($terrain->type) {
                    case 'Terrains taxables':
                        return '<span style="color: #ff7600">Terrains taxables</span>';
                        break;             

                    case 'Exonerations permanentes':
                        return '<span style="color: #595858">Exonérations permanentes</span>';
                        break;

                    case 'Exonerations temporaires':
                        return '<span style="color: #39AFC0">Exonérations temporaires</span>';
                        break;

                    case 'Terrains supprimes':
                        return '<span style="color: #F36281">Terrains aménagés/construits</span>';
                        break;

                    default:
                        return '<b style="color:gray">Néant</b>';
                        break;
                }
                return $terrain->type;
            })

            ->addColumn('declare', function ($terrain) {
                $has_declaration = Declaration::where('id_terrain', $terrain->id)
                    ->whereYear('date', date('Y'))
                    ->exists();

                if ($has_declaration) {
                    return '<span style="color: #558B2F">Déclaré</span>';
                } else {
                    return '<span style="color: #EF5350">Non déclaré</span>';
                }
            })

            ->addColumn('liquidation', function ($terrain) {
                $has_declaration = Liquidation::where('id_terrain', $terrain->id)
                    ->whereYear('date', date('Y'))
                    ->exists();

                if ($has_declaration) {
                    return '<span style="color: #558B2F">Liquidé</span>';
                } else {
                    return '<span style="color: #EF5350">Non liquidé</span>';
                }
            })


            // ->addColumn('liquidation', function ($terrain) {
            //     $currentYear = now()->year;
            //     $liquide_count = Liquidation::where('id_terrain', $terrain->id)
            //         ->whereYear('created_at', $currentYear)
            //         ->where('paid', 1)        
            //         ->count();

            //     // Determine liquidation status based on the count
            //     if ($liquide_count > 0) {
            //         return '<span style="color: #558B2F">Liquidé</span>';
            //     } else {
            //         return '<span style="color: #EF5350">Non liquidé</span>';
            //     }
            // })
            ->rawColumns(['article', 'ref', 'surface', 'zones', 'type', 'proprietaires', 'declare', 'liquidation'])
            ->only(['article', 'ref', 'surface', 'statut_foncier', 'adresse', 'zones', 'type', 'proprietaires', 'declare', 'liquidation'])
            ->make(true);
    }


    public function terrains_taxables_datatable(Request $request)
    {
        $filters = json_decode($request->filters);
        $terrains = Terrain::with(['zones', 'proprietaires', 'zones_pa'])->where('type', "Terrains taxables")->orderBy('created_at', 'desc');

        foreach ($filters as $filter) {

            if (isset($filter->params)) {
                $params = explode(',', $filter->params);
                foreach ($params as $param) {
                    if ($param == "remove_spaces") {
                        $filter->value = str_replace(' ', '', $filter->value);
                    }
                }
            }


            if ($filter->column === 'search') {
                $searchValue = $filter->value;
                $terrains = $terrains->where(function ($query) use ($searchValue) {
                    $query->where('ref', 'LIKE', '%' . $searchValue . '%')
                        ->orWhereHas('proprietaires', function ($subQuery) use ($searchValue) {
                            $subQuery->where('type', 'morale')
                                ->where('rc', 'LIKE', '%' . $searchValue . '%');
                        })
                        ->orWhereHas('proprietaires', function ($subQuery) use ($searchValue) {
                            $subQuery->where('type', 'physique')
                                ->where('cine', 'LIKE', '%' . $searchValue . '%');
                        });
                });
                continue;
            }



            if ($filter->column == 'zone') {
                $terrains = $terrains->whereHas('zones', function ($query) use ($filter) {
                    $query->where('zones.id', $filter->value);
                });
                continue;
            }

            if ($filter->column == 'proprietaire_type') {
                $terrains = $terrains->whereHas('proprietaires', function ($query) use ($filter) {
                    $query->where('type', $filter->value);
                });
                continue;
            }

            if ($filter->column == 'qualite_redevable') {
                $terrains = $terrains->whereHas('proprietaires.qualites', function ($query) use ($filter) {
                    $query->where('qualite', $filter->value);
                });
                continue;
            }


            if ($filter->column === 'declaration') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }

            if ($filter->column === 'liquidation') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('liquidations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }

            // if ($filter->column == 'created_at') {

            //     $from =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[0]));
            //     $to =  Carbon::createFromFormat('d/m/Y', trim(explode('-', $filter->value)[1]));

            //     $terrains = $terrains->whereBetween($filter->column, [$from, $to]);
            // } else {
            //     if ($filter->table == '') {
            //         $terrains = $terrains->where($filter->column, '=', $filter->value);
            //     } else {
            //         $terrains = $terrains->whereHas($filter->table, function ($query) use ($filter) {
            //             $query->where($filter->table . '.' . $filter->column, '=', $filter->value);
            //         });
            //     }
            // }
        }

        return Datatables::eloquent($terrains)
            ->editColumn('article', function ($terrain) {
                return '<i style="font-size: 0.8rem;color: #7C7C7C;" class="fa-regular fa-map-marker-alt"></i>&nbsp<a  href="' . asset('') . 'terrains/show/' . $terrain->id . '">' . $terrain->article . '</a>&nbsp';
            })
            ->addColumn('ref', function ($terrain) {
                return $terrain->ref;
            })
            ->addColumn('surface', function ($terrain) {
                return number_format($terrain->surface, 0, '.', ' ') . 'm²';
            })
            ->addColumn('statut_foncier', function ($terrain) {
                return $terrain->statut_foncier;
            })

            ->addColumn('adresse', function ($terrain) {
                return $terrain->adresse;
            })


            ->addColumn('zones', function ($terrain) {
                $currentYear = date('Y');
                $zones = $terrain->zones->where('annee', $currentYear);

                if ($zones->count() > 0) {
                    return $zones->pluck('nom')->implode(', ');
                } else {
                    return '<b style="color:gray">Néant</b>';
                }
            })
            // ->addColumn('zones_pa', function ($terrain) {
            //     $zones_pa = $terrain->zones_pa;

            //     if ($zones_pa->count() > 0) {
            //         return $zones_pa->pluck('nom')->implode(', ');
            //     } else {
            //         return '<b style="color:gray">Néant</b>';
            //     }
            // })
            ->addColumn('proprietaires', function ($terrain) {
                $proprietaires = $terrain->proprietaires;
                $html = '<b style="color:gray">Néant</b>';
                if ($proprietaires->count() > 0) {
                    $html = '<select class="form-control select2nc">';
                    foreach ($proprietaires as $prop) {
                        $typeInfo = '';
                        if ($prop->type === 'physique') {
                            $typeInfo = ' | CINE: ' . ($prop->cine ?? 'N/A');
                        } elseif ($prop->type === 'morale') {
                            $typeInfo = ' | RC: ' . ($prop->rc ?? 'N/A') . ' | Ville: ' . ($prop->ville_rc ?? 'N/A');
                        }
                        $html .= '<option>' . $prop->nom . ' ' . $prop->prenom . $typeInfo . '</option>';
                    }
                    $html .= '</select>';
                }
                return $html;
            })
            ->addColumn('type', function ($terrain) {
                switch ($terrain->type) {
                    case 'Terrains taxables':
                        return '<span style="color: #ff7600">Terrains taxables</span>';
                        break;

                    case 'Terrains non taxables':
                        return '<span style="color: #558B2F">Terrains non taxables</span>';
                        break;

                    case 'Exonerations permanentes':
                        return '<span style="color: #595858">Exonérations permanentes</span>';
                        break;

                    case 'Exonerations temporaires':
                        return '<span style="color: #39AFC0">Exonérations temporaires</span>';
                        break;

                    case 'Terrains supprimes':
                        return '<span style="color: #F36281">Terrains aménagés/construits</span>';
                        break;

                    default:
                        return '<b style="color:gray">Néant</b>';
                        break;
                }
                return $terrain->type;
            })

            ->addColumn('declare', function ($terrain) {
                $has_declaration = Declaration::where('id_terrain', $terrain->id)
                    ->whereYear('date', date('Y'))
                    ->exists();

                if ($has_declaration) {
                    return '<span style="color: #558B2F">Déclaré</span>';
                } else {
                    return '<span style="color: #EF5350">Non déclaré</span>';
                }
            })

            ->addColumn('liquidation', function ($terrain) {
                $has_declaration = Liquidation::where('id_terrain', $terrain->id)
                    ->whereYear('date', date('Y'))
                    ->exists();

                if ($has_declaration) {
                    return '<span style="color: #558B2F">Liquidé</span>';
                } else {
                    return '<span style="color: #EF5350">Non liquidé</span>';
                }
            })


            // ->addColumn('liquidation', function ($terrain) {
            //     $currentYear = now()->year;
            //     $liquide_count = Liquidation::where('id_terrain', $terrain->id)
            //         ->whereYear('created_at', $currentYear)
            //         ->where('paid', 1)        
            //         ->count();

            //     // Determine liquidation status based on the count
            //     if ($liquide_count > 0) {
            //         return '<span style="color: #558B2F">Liquidé</span>';
            //     } else {
            //         return '<span style="color: #EF5350">Non liquidé</span>';
            //     }
            // })
            ->rawColumns(['article', 'ref', 'surface', 'zones', 'proprietaires', 'declare', 'liquidation'])
            ->only(['article', 'ref', 'surface', 'statut_foncier', 'adresse', 'zones', 'proprietaires', 'declare', 'liquidation'])
            ->make(true);
    }



    public function terrains_exonerations_temporaires_datatable(Request $request)
    {
        $filters = json_decode($request->filters);
        $terrains = Terrain::with(['zones', 'proprietaires', 'zones_pa'])->where('type', "Exonerations temporaires")->orderBy('created_at', 'desc');

        foreach ($filters as $filter) {

            if (isset($filter->params)) {
                $params = explode(',', $filter->params);
                foreach ($params as $param) {
                    if ($param == "remove_spaces") {
                        $filter->value = str_replace(' ', '', $filter->value);
                    }
                }
            }

            if ($filter->column === 'search') {
                $searchValue = $filter->value;
                $terrains = $terrains->where(function ($query) use ($searchValue) {
                    $query->where('ref', 'LIKE', '%' . $searchValue . '%')
                        ->orWhereHas('proprietaires', function ($subQuery) use ($searchValue) {
                            $subQuery->where('type', 'morale')
                                ->where('rc', 'LIKE', '%' . $searchValue . '%');
                        })
                        ->orWhereHas('proprietaires', function ($subQuery) use ($searchValue) {
                            $subQuery->where('type', 'physique')
                                ->where('cine', 'LIKE', '%' . $searchValue . '%');
                        });
                });
                continue;
            }


            if ($filter->column == 'zone') {
                $terrains = $terrains->whereHas('zones', function ($query) use ($filter) {
                    $query->where('zones.id', $filter->value);
                });
                continue;
            }

            if ($filter->column == 'proprietaire_type') {
                $terrains = $terrains->whereHas('proprietaires', function ($query) use ($filter) {
                    $query->where('type', $filter->value);
                });
                continue;
            }

            if ($filter->column == 'qualite_redevable') {
                $terrains = $terrains->whereHas('proprietaires.qualites', function ($query) use ($filter) {
                    $query->where('qualite', $filter->value);
                });
                continue;
            }


            if ($filter->column === 'declaration') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }
        }

        return Datatables::eloquent($terrains)
            ->editColumn('article', function ($terrain) {
                return '<i style="font-size: 0.8rem;color: #7C7C7C;" class="fa-regular fa-map-marker-alt"></i>&nbsp<a  href="' . asset('') . 'terrains/show/' . $terrain->id . '">' . $terrain->article . '</a>&nbsp';
            })
            ->addColumn('ref', function ($terrain) {
                return $terrain->ref;
            })
            ->addColumn('surface', function ($terrain) {
                return number_format($terrain->surface, 0, '.', ' ') . 'm²';
            })
            ->addColumn('statut_foncier', function ($terrain) {
                return $terrain->statut_foncier;
            })

            ->addColumn('adresse', function ($terrain) {
                return $terrain->adresse;
            })


            ->addColumn('zones', function ($terrain) {
                $currentYear = date('Y');
                $zones = $terrain->zones->where('annee', $currentYear);

                if ($zones->count() > 0) {
                    return $zones->pluck('nom')->implode(', ');
                } else {
                    return '<b style="color:gray">Néant</b>';
                }
            })
            // ->addColumn('zones_pa', function ($terrain) {
            //     $zones_pa = $terrain->zones_pa;

            //     if ($zones_pa->count() > 0) {
            //         return $zones_pa->pluck('nom')->implode(', ');
            //     } else {
            //         return '<b style="color:gray">Néant</b>';
            //     }
            // })
            ->addColumn('proprietaires', function ($terrain) {
                $proprietaires = $terrain->proprietaires;
                $html = '<b style="color:gray">Néant</b>';
                if ($proprietaires->count() > 0) {
                    $html = '<select class="form-control select2nc">';
                    foreach ($proprietaires as $prop) {
                        $typeInfo = '';
                        if ($prop->type === 'physique') {
                            $typeInfo = ' | CINE: ' . ($prop->cine ?? 'N/A');
                        } elseif ($prop->type === 'morale') {
                            $typeInfo = ' | RC: ' . ($prop->rc ?? 'N/A') . ' | Ville: ' . ($prop->ville ?? 'N/A');
                        }
                        $html .= '<option>' . $prop->nom . ' ' . $prop->prenom . $typeInfo . '</option>';
                    }
                    $html .= '</select>';
                }
                return $html;
            })

            ->addColumn('declare', function ($terrain) {
                $has_declaration = Declaration::where('id_terrain', $terrain->id)
                    ->whereYear('date', date('Y'))
                    ->exists();

                if ($has_declaration) {
                    return '<span style="color: #558B2F">Déclaré</span>';
                } else {
                    return '<span style="color: #EF5350">Non déclaré</span>';
                }
            })


            ->addColumn('delai', function ($terrain) {
                // Find TT_ET operation
                $ttOperation = $terrain->operations
                    ->where('operation_type', 'TT_ET')
                    ->sortByDesc('date')
                    ->first();

                if (!$ttOperation) {
                    return '<b style="color:gray">Aucune opération</b>';
                }

                $delai = $this->calculateDelai($ttOperation->operation_motif, $terrain->surface);             

                if ($delai == 0) {
                    return '<b style="color:gray">Motif non reconnu</b>';
                }

                $operationDate = Carbon::parse($ttOperation->date);
                $expiryDate = $operationDate->copy()->addYears($delai);
                $currentDate = Carbon::now();

              
                if ($currentDate->gt($expiryDate)) {
                    $expiredSince = (int) $currentDate->diffInDays($expiryDate);
                    return '<span style="color: #EF5350; font-weight: bold;">Expiré depuis ' . $expiredSince . ' jours</span>';
                } else {
                    $remainingDays = (int) $currentDate->diffInDays($expiryDate);
                    $remainingYears = round($remainingDays / 365.25, 1);
                    
                    $color = '#4CAF50'; 
                    if ($remainingDays <= 60) { 
                        $color = '#EF5350'; 
                    } elseif ($remainingDays <= 90) { 
                        $color = '#FF9800'; 
                    } elseif ($remainingDays <= 180) { 
                        $color = '#FFC107'; 
                    }

                    return '<span style="color: ' . $color . '; font-weight: bold;">Délai restant ' . $remainingYears . 'ans (' . $remainingDays . 'j)</span>';
                }
            })


            ->rawColumns(['article', 'ref', 'surface', 'zones', 'proprietaires', 'declare','delai'])
            ->only(['article', 'ref', 'surface', 'statut_foncier', 'adresse', 'zones', 'proprietaires', 'declare','delai'])
            ->make(true);
    }



    public function terrains_supprimes_datatable(Request $request)
    {
        $filters = json_decode($request->filters);
        $terrains = Terrain::with(['zones', 'proprietaires', 'zones_pa'])->where('type', "Terrains supprimes")->orderBy('created_at', 'desc');

        foreach ($filters as $filter) {

            if (isset($filter->params)) {
                $params = explode(',', $filter->params);
                foreach ($params as $param) {
                    if ($param == "remove_spaces") {
                        $filter->value = str_replace(' ', '', $filter->value);
                    }
                }
            }

            if ($filter->column === 'search') {
                $searchValue = $filter->value;
                $terrains = $terrains->where(function ($query) use ($searchValue) {
                    $query->where('ref', 'LIKE', '%' . $searchValue . '%')
                        ->orWhereHas('proprietaires', function ($subQuery) use ($searchValue) {
                            $subQuery->where('type', 'morale')
                                ->where('rc', 'LIKE', '%' . $searchValue . '%');
                        })
                        ->orWhereHas('proprietaires', function ($subQuery) use ($searchValue) {
                            $subQuery->where('type', 'physique')
                                ->where('cine', 'LIKE', '%' . $searchValue . '%');
                        });
                });
                continue;
            }


            if ($filter->column == 'proprietaire_type') {
                $terrains = $terrains->whereHas('proprietaires', function ($query) use ($filter) {
                    $query->where('type', $filter->value);
                });
                continue;
            }

            if ($filter->column == 'qualite_redevable') {
                $terrains = $terrains->whereHas('proprietaires.qualites', function ($query) use ($filter) {
                    $query->where('qualite', $filter->value);
                });
                continue;
            }


            if ($filter->column === 'declaration') {
                if ($filter->value === 'oui') {
                    $terrains = $terrains->whereHas('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                } elseif ($filter->value === 'non') {
                    $terrains = $terrains->whereDoesntHave('declarations', function ($query) {
                        $query->whereYear('date', date('Y'));
                    });
                }
                continue;
            }
        }

        return Datatables::eloquent($terrains)
            ->editColumn('article', function ($terrain) {
                return '<i style="font-size: 0.8rem;color: #7C7C7C;" class="fa-regular fa-map-marker-alt"></i>&nbsp<a  href="' . asset('') . 'terrains/show/' . $terrain->id . '">' . $terrain->article . '</a>&nbsp';
            })
            ->addColumn('ref', function ($terrain) {
                return $terrain->ref;
            })
            ->addColumn('surface', function ($terrain) {
                return number_format($terrain->surface, 0, '.', ' ') . 'm²';
            })
            ->addColumn('statut_foncier', function ($terrain) {
                return $terrain->statut_foncier;
            })

            ->addColumn('adresse', function ($terrain) {
                return $terrain->adresse;
            })


            ->addColumn('zones', function ($terrain) {
                $currentYear = date('Y');
                $zones = $terrain->zones->where('annee', $currentYear);

                if ($zones->count() > 0) {
                    return $zones->pluck('nom')->implode(', ');
                } else {
                    return '<b style="color:gray">Néant</b>';
                }
            })
            // ->addColumn('zones_pa', function ($terrain) {
            //     $zones_pa = $terrain->zones_pa;

            //     if ($zones_pa->count() > 0) {
            //         return $zones_pa->pluck('nom')->implode(', ');
            //     } else {
            //         return '<b style="color:gray">Néant</b>';
            //     }
            // })
            ->addColumn('proprietaires', function ($terrain) {
                $proprietaires = $terrain->proprietaires;
                $html = '<b style="color:gray">Néant</b>';
                if ($proprietaires->count() > 0) {
                    $html = '<select class="form-control select2nc">';
                    foreach ($proprietaires as $prop) {
                        $typeInfo = '';
                        if ($prop->type === 'physique') {
                            $typeInfo = ' | CINE: ' . ($prop->cine ?? 'N/A');
                        } elseif ($prop->type === 'morale') {
                            $typeInfo = ' | RC: ' . ($prop->rc ?? 'N/A') . ' | Ville: ' . ($prop->ville ?? 'N/A');
                        }
                        $html .= '<option>' . $prop->nom . ' ' . $prop->prenom . $typeInfo . '</option>';
                    }
                    $html .= '</select>';
                }
                return $html;
            })



            ->rawColumns(['article', 'ref', 'surface', 'zones', 'proprietaires'])
            ->only(['article', 'ref', 'surface', 'statut_foncier', 'adresse', 'zones', 'proprietaires'])
            ->make(true);
    }


    public function show(Terrain $terrain)
    {
        $motif_exoneration = null;
        $type_motif = null;
        $avancement_travaux = null;
        $date_avancement_travaux = null;
        $zones_pa = ZonePA::all();

        $terrain->load([
            'zones' => function ($query) {
                $query->where('annee', date('Y'));
            },
            'proprietaires' => function ($query) {
                $query->withTrashed();
            },
            'proprietaires.proprietaire_documents',
            'zones_pa',
            'operations' => function ($query) {
                $query->orderBy('created_at', 'desc');
            },
            'liquidations' => function ($query) {
                $query->whereYear('date', date('Y'));
            }
        ]);

        if ($terrain->type == "Exonerations temporaires" || $terrain->type == "Terrains supprimes") {
            $exonerationOperation = TerrainOperation::where('terrain_id', $terrain->id)
                ->where('operation_type', 'TT_ET')
                ->first();


            if ($exonerationOperation) {
                $motif_exoneration = $exonerationOperation->operation_motif;

                if ($exonerationOperation->operation_data) {
                    $type_motif = $exonerationOperation->type;
                }

                $avancementOperation = TerrainOperation::where('terrain_id', $terrain->id)
                    ->where('operation_type', 'AVANCEMENT_TRAVAUX')
                    ->orderBy('date', 'desc')
                    ->first();
                if ($avancementOperation && $avancementOperation->operation_motif) {
                    $avancement_travaux = $avancementOperation->operation_motif;
                    $date_avancement_travaux = $avancementOperation->date;
                }
            }
        }

        $LIQUIDATION_LOCKED =  !$terrain->specific_document('DECLARATION_PDF');


        $derniere_declaration = TerrainOperation::where('terrain_id', $terrain->id)
            ->where('operation_type', 'DEC_TERRAIN')
            ->orderBy('date', 'desc')
            ->first();


        $derniere_liquidation = TerrainOperation::where('terrain_id', $terrain->id)
            ->where('operation_type', 'LIQ_TERRAIN')
            ->orderBy('date', 'desc')
            ->first();

        $proprietaire_documents_list = [];

        $tarif = $terrain->zones->first()->tarif ?? 0;
        $surface = $terrain->surface;
        // calculate the taxable surface 
        $surface_taxable = $terrain->zones_pa
        ->where('taxability', 1)
        ->sum(function ($zone_pa) use ($terrain) {
            return $terrain->surface * $zone_pa->pivot->ratio;
        });


        $active_proprietaires = $terrain->proprietaires->whereNull('deleted_at');
        // $old_proprietaires = $terrain->proprietaires->whereNotNull('deleted_at');
        $old_proprietaires = $terrain->old_proprietaires()->get();

        $liquidations = [];

        //get the last 4 years
        $lastPaid = Liquidation::where('id_terrain', $terrain->id)
            ->orderByDesc('date')
            ->first();
        $startYear = $lastPaid
            ? Carbon::parse($lastPaid->date)->year + 1
            : now()->subYears(3)->year;
        $currentYear = now()->year;

        $montant_initial = 0;
        $montant_sanctions = 0;
        for ($year = $startYear; $year <= $currentYear; $year++) {
            $montant_tnb = $surface_taxable * $tarif;
            $sanctions = $this->calculateSanctions($year, $montant_tnb);
            $montant_initial += $montant_tnb;
            $montant_sanctions += $sanctions['total_sanctions'];
            $liquidations[] = [
                'year' => $year,
                'montant_tnb' => $montant_tnb,
                'sanctions' => $sanctions['details'],
                'total' => $montant_tnb + $sanctions['total_sanctions'],
                'months_passed' => $sanctions['months_passed'],
            ];
        }
        $montant_total = $montant_initial + $montant_sanctions;
        $declaration_documents_list = [
            [
                'type_document' => 'PLAN_CADASTRAL_SITUATION_PDF',
                'name' => "Plan cadastral / Plan situation (PDF)",
                'document' => $terrain->specific_document('PLAN_CADASTRAL_SITUATION_PDF'),
                'file_type' => 'pdf',
                'w_roles' => 'all',
                'r_roles' => 'all',
                'active' => 1,
                'validation' => 1,
                'date_doc' => 0
            ],
            [
                'type_document' => 'CALCUL_CONTENANCE_PDF',
                'name' => "Calcul de contenance (PDF)",
                'document' => $terrain->specific_document('CALCUL_CONTENANCE_PDF'),
                'file_type' => 'pdf',
                'w_roles' => 'all',
                'r_roles' => 'all',
                'active' => 1,
                'validation' => 1,
                'date_doc' => 0
            ],
            [
                'type_document' => 'CERTIFICAT_ATTESTATION_PROPRIETE_PDF',
                'name' => "Certificat de propriété (PDF)",
                'document' => $terrain->specific_document('CERTIFICAT_ATTESTATION_PROPRIETE_PDF'),
                'file_type' => 'pdf',
                'w_roles' => 'all',
                'r_roles' => 'all',
                'active' => 1,
                'validation' => 1,
                'date_doc' => 0
            ]

        ];


        $declaration_document = [
            'type_document' => 'DECLARATION_PDF',
            'name' => "Déclaration du terrain",
            'document' => $terrain->specific_document('DECLARATION_PDF'),
            'file_type' => 'pdf',
            'w_roles' => 'all',
            'r_roles' => 'all',
            'active' => 1,
            'validation' => 1,
            'date_doc' => 1
        ];


        $declaration_changement_proprietaire_document = [
            'type_document' => 'DECLARATION_CHANGEMENT_PROPRIETAIRE_PDF',
            'name' => "Déclaration changement de propriétaire",
            'document' => $terrain->specific_document('DECLARATION_CHANGEMENT_PROPRIETAIRE_PDF'),
            'file_type' => 'pdf',
            'w_roles' => 'all',
            'r_roles' => 'all',
            'active' => 1,
            'validation' => 1,
            'date_doc' => 1
        ];


        $exoneration_temporaire_document = [
            'type_document' => 'AUTORISATION_EXONERATION_TEMPORAIRE_PDF',
            'name' => "L'autorisation",
            'document' => $terrain->specific_document('AUTORISATION_EXONERATION_TEMPORAIRE_PDF'),
            'file_type' => 'pdf',
            'w_roles' => 'all',
            'r_roles' => 'all',
            'active' => 1,
            'validation' => 1,
            'date_doc' => 1
        ];


        $liquidation_document = [
            'type_document' => 'LIQUIDATION_PDF',
            'name' => "Liquidation du terrain",
            'document' => $terrain->specific_document('LIQUIDATION_PDF'),
            'file_type' => 'pdf',
            'w_roles' => 'all',
            'r_roles' => 'all',
            'active' => 1,
            'validation' => 1,
            'date_doc' => 1
        ];


        $avancement_travaux_document = [
            'type_document' => 'AVANCEMENT_TRAVAUX_PDF',
            'name' => "Avancement des travaux (PDF)",
            'document' => $terrain->specific_document('AVANCEMENT_TRAVAUX_PDF'),
            'file_type' => 'pdf',
            'w_roles' => 'all',
            'r_roles' => 'all',
            'active' => 1,
            'validation' => 1,
            'date_doc' => 0
        ];

        $terrain_phototheque = [

            [
                'type_document' => 'TERRAIN_NU_PHOTO',
                'name' => "Terrain nu photo",
                'title' => "",
                'photos' => $terrain->specific_photos('TERRAIN_NU_PHOTO'),
                'file_type' => 'png jpeg jpg bmp'
            ],
        ];

        $latest_three_photos = Terrain_document::where([
            ['active', 1],
            ['type_document', "TERRAIN_NU_PHOTO"],
            ['terrain_id', $terrain->id]
        ])
            ->orderBy('created_at', 'desc')
            ->take(3)
            ->get();


        return view('terrains.show', [
            'zones_pa' => $zones_pa,
            'terrain' => $terrain,
            'motif_exoneration' => $motif_exoneration,
            'type_motif_exoneration' => $type_motif,
            'avancement_travaux' => $avancement_travaux,
            'date_avancement_travaux' => $date_avancement_travaux,
            'LIQUIDATION_LOCKED' => $LIQUIDATION_LOCKED,
            'active_proprietaires' => $active_proprietaires,
            'old_proprietaires' => $old_proprietaires,
            'liquidations' => $liquidations,
            'montant_initial' => $montant_initial,
            'montant_sanctions' => $montant_sanctions,
            'montant_total' => $montant_total,
            'proprietaire_documents_list' => $proprietaire_documents_list,
            'declaration_documents_list' => $declaration_documents_list,
            'declaration_document' => $declaration_document,
            'liquidation_document' => $liquidation_document,
            'declaration_changement_proprietaire_document' => $declaration_changement_proprietaire_document,
            'exoneration_temporaire_document' => $exoneration_temporaire_document,
            'terrain_phototheque' => $terrain_phototheque,
            'avancement_travaux_document' => $avancement_travaux_document,
            'derniere_declaration' => $derniere_declaration,
            'derniere_liquidation' => $derniere_liquidation,
            'latest_three_photos' => $latest_three_photos,
        ]);
    }





    public function store(Request $request)
    {
        $validated_data = $request->validate([
            'id' => 'required|exists:terrains,id',
            'proprietaires' => 'required|json',
        ]);

        $terrain_id = $validated_data['id'];
        $proprietaires_array = json_decode($request->proprietaires, true);
        $documents = json_decode($request->documents);
        $documents_to_delete = json_decode($request->documents_to_delete);

        foreach ($documents as $document) {
            $terrain_document = Terrain_document::where('id', $document->id)->get()->first();

            $terrain_document->terrain_id = $terrain_id;
            $terrain_document->save();

            if ($terrain_document->type_document == "DECLARATION_PDF") {
                $operation = new TerrainOperation();
                $operation->id = Str::uuid();
                $operation->terrain_id = $terrain_id;
                $operation->operation_num = $terrain_document->numero;
                $operation->user_id = auth()->id();

                $operation->operation_type = "DEC_TERRAIN";
                $operation->date = now()->toDateString();
                $operation->operation_data = [
                    "document_id" => $terrain_document->id,
                    "filename" => $terrain_document->filename,
                    "type_document" => $terrain_document->type_document
                ];


                $operation->decision = "valider";
                $operation->related_model_type = get_class($terrain_document);
                $operation->related_model_id = $terrain_document->id;
                $operation->save();

                //add new declaration
                $declaration = new Declaration();
                $declaration->id = Str::uuid();
                $declaration->id_terrain = $terrain_id;
                $declaration->numero_declaration = $terrain_document->numero;
                $declaration->date = $terrain_document->date_document;
                $declaration->save();
            }

            if ($terrain_document->type_document == "DECLARATION_CHANGEMENT_PROPRIETAIRE_PDF") {
                $operation = new TerrainOperation();
                $operation->id = Str::uuid();
                $operation->terrain_id = $terrain_id;
                $operation->operation_num = $terrain_document->numero;
                $operation->user_id = auth()->id();

                $operation->operation_type = "DEC_CHANGEMENT_PROP";
                $operation->date = now()->toDateString();
                $operation->operation_data = [
                    "document_id" => $terrain_document->id,
                    "filename" => $terrain_document->filename,
                    "type_document" => $terrain_document->type_document
                ];


                $operation->decision = "valider";
                $operation->related_model_type = get_class($terrain_document);
                $operation->related_model_id = $terrain_document->id;
                $operation->save();               
            }

            if ($terrain_document->type_document == "LIQUIDATION_PDF") {
                $operation = new TerrainOperation();
                $operation->id = Str::uuid();
                $operation->terrain_id = $terrain_id;
                $operation->operation_num = $terrain_document->numero;
                $operation->user_id = auth()->id();

                $operation->operation_type = "LIQ_TERRAIN";
                $operation->date = now()->toDateString();
                $operation->operation_data = [
                    "document_id" => $terrain_document->id,
                    "filename" => $terrain_document->filename,
                    "type_document" => $terrain_document->type_document
                ];


                $operation->decision = "valider";
                $operation->related_model_type = get_class($terrain_document);
                $operation->related_model_id = $terrain_document->id;
                $operation->save();

                //add new declaration
                $liquidation = new Liquidation();
                $liquidation->id = Str::uuid();
                $liquidation->id_terrain = $terrain_id;
                $liquidation->numero = $terrain_document->numero;
                $liquidation->date = $terrain_document->date_document;
                $liquidation->montant = floatval(str_replace(' MAD', '', $terrain_document->montant_liquidation));
                $liquidation->save();
            }


            if ($terrain_document->type_document == "AUTORISATION_EXONERATION_TEMPORAIRE_PDF") {
                $operation = new TerrainOperation();
                $operation->id = Str::uuid();
                $operation->terrain_id = $terrain_id;
                $operation->operation_num = $terrain_document->numero;
                $operation->user_id = auth()->id();
                $operation->operation_motif = $request->motif_exoneration_temporaire;

                $operation->operation_type = "TT_ET";
                $operation->date = now()->toDateString();
                $operationData = [
                    "document_id" => $terrain_document->id,
                    "filename" => $terrain_document->filename,
                    "type_document" => $terrain_document->type_document,
                ];

                // Add conditional data based on motif_exoneration_temporaire
                if ($request->motif_exoneration_temporaire == "exploitation_professionnelle") {
                    $operationData["type"] = $request->type_motif_exploitation_prof;
                    $operationData["surface_exoneration"] = $request->surface_exoneration;
                } elseif ($request->motif_exoneration_temporaire == "construction") {
                    $operationDatat["type"] = $request->type_motif;
                }

                // $operation->operation_data = [
                //     "document_id" => $terrain_document->id,
                //     "filename" => $terrain_document->filename,
                //     "type_document" => $terrain_document->type_document,
                //     "type_construction" => $request->type_construction,
                // ];

                $operation->operation_data = $operationData;
                $operation->decision = "valider";
                $operation->related_model_type = get_class($terrain_document);
                $operation->related_model_id = $terrain_document->id;
                $operation->save();

                //change terrain type
                $terrain_update_type = Terrain::where('id', $terrain_id)->first();
                $terrain_update_type->type = "Exonerations temporaires";
                $terrain_update_type->save();
            }
        }


        foreach ($documents_to_delete as $document) {
            $dossier_document = Terrain_document::where('id', $document)->get()->first();
            $dossier_document->delete();
        }


        // AVANCEMENT TRAVAUX SECTION
        // if ($request->has('avancement_travaux') && $request->has('date_avancement_travaux')) {
        if ($terrain_document->type_document == "AVANCEMENT_TRAVAUX_PDF") {
            $avancement_travaux = $request->avancement_travaux;
            $date_avancement = $request->date_avancement_travaux;

            $terrain = Terrain::findOrFail($terrain_id);

            if ($terrain->type === "Exonerations temporaires" && !empty($avancement_travaux) && !empty($date_avancement)) {

                $operation = new TerrainOperation();
                $operation->id = Str::uuid();
                $operation->terrain_id = $terrain_id;
                // $operation->operation_num = 'AVT-' . date('YmdHis');
                $operation->user_id = auth()->id();
                $operation->operation_type = "AVANCEMENT_TRAVAUX";
                $operation->operation_motif = $avancement_travaux;
                $operation->date = Carbon::createFromFormat('d/m/Y', $date_avancement)->format('Y-m-d');
                $operation->operation_data = [
                    "avancement_type" => $avancement_travaux,
                    "date_avancement" => $date_avancement,
                    "document_id" => $terrain_document->id,
                    "filename" => $terrain_document->filename,
                    "type_document" => $terrain_document->type_document,
                ];
                $operation->decision = "valider";
                $operation->related_model_type = get_class($terrain);
                $operation->related_model_id = $terrain->id;
                $operation->save();


                if ($avancement_travaux === "achevement") {
                    $terrain->type = "Terrains supprimes";
                    $terrain->save();
                } elseif ($avancement_travaux === "non-achevement") {
                    $terrain->type = "Terrains taxables";
                    $terrain->save();
                }
            }
        }
        // END OF AVANCEMENT TRAVAUX SECTION


        // Validate total quote part sum
        $total_quote_part = array_sum(array_map(function ($proprietaire) {
            if (isset($proprietaire['quote_part'])) {
                return (float) $proprietaire['quote_part'];
            } elseif (isset($proprietaire['pivot']['quote_part'])) {
                return (float) $proprietaire['pivot']['quote_part'];
            }
            return 0;
        }, $proprietaires_array));

        if (bccomp((string) $total_quote_part, '1', 2) !== 0) {
            return response()->json([
                'success' => 0,
                'error' => 'La somme des quotes-parts doit être égale à 1.',
            ], 400);
        }

        $terrain = Terrain::with('proprietaires')->findOrFail($terrain_id);
        $existing_proprietaires = $terrain->proprietaires;


        foreach ($existing_proprietaires as $existing_proprietaire) {
            $exist = false;

            foreach ($proprietaires_array as $proprietaire_data) {
                if ($existing_proprietaire->id == ($proprietaire_data['id'] ?? null)) {
                    $exist = true;
                }
            }

            if (!$exist) {
                DB::table('proprietaire_terrain')
                    ->where('terrain_id', $terrain->id)
                    ->where('proprietaire_id', $existing_proprietaire->id)
                    ->update([
                        'active' => 0,
                        'updated_at' => now(),
                    ]);
            }
        }

        // Add or update propriétaires
        foreach ($proprietaires_array as $proprietaire_data) {
            $proprietaire_id = $proprietaire_data['id'] ?? Str::uuid();
            $proprietaire_type = $proprietaire_data['type'] ?? 'physique';
            // Check if the propriétaire exists or needs to be created
            $proprietaire = Proprietaire::firstOrNew(
                ['id' => $proprietaire_id],
                [
                    'type' => $proprietaire_type,
                    'nom' => $proprietaire_data['nom'] ?? '',
                    'prenom' => $proprietaire_data['prenom'] ?? '',
                    'cine' => $proprietaire_data['cine'] ?? '',
                    'rc' => $proprietaire_data['rc'] ?? '',
                    'ville_rc' => $proprietaire_data['ville_rc'] ?? '',
                ]
            );
            $proprietaire->save();

            $existing_pivot_entry = DB::table('proprietaire_terrain')
                ->where('terrain_id', $terrain->id)
                ->where('proprietaire_id', $proprietaire->id)
                ->first();

            if ($existing_pivot_entry) {
                if (isset($proprietaire_data['quote_part'])) {
                    DB::table('proprietaire_terrain')
                        ->where('id', $existing_pivot_entry->id)
                        ->update([
                            'quote_part' => (float)$proprietaire_data['quote_part'],
                            'active' => 1,
                            'updated_at' => now(),
                        ]);
                }
            } else {
                if (isset($proprietaire_data['quote_part'])) {
                    DB::table('proprietaire_terrain')->insert([
                        'id' => Str::uuid(),
                        'terrain_id' => $terrain->id,
                        'proprietaire_id' => $proprietaire->id,
                        'quote_part' => (float)$proprietaire_data['quote_part'],
                        'active' => 1,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
        }

        return response()->json(['success' => true]);
    }


    public function terrain_data(string $terrain_id)
    {
        $terrain = Terrain::where('id', $terrain_id)->get()->first();

        $terrain->format_created_at = Carbon::createFromFormat('Y-m-d H:i:s',  $terrain->created_at)->format('d/m/Y');

        $terrain->etat_avancement_color = $terrain->etat_avancement_color('');
        return  $terrain;
    }

    public function assujettissement(Request $request)
    {

        if ($request->filled('date_operation')) {
            try {
                $date = \Carbon\Carbon::createFromFormat('d/m/Y', $request->date_operation);
                $request->merge(['date_operation' => $date->format('Y-m-d')]);
            } catch (\Exception $e) {
                return response()->json(['success' => 0, 'message' => 'Date invalide'], 422);
            }
        }

        $request->validate([
            'operation_num' => 'nullable|string|max:255',
            'operation_motif' => 'nullable|string',
            'date_operation' => 'nullable|date',
            'type_construction' => 'nullable|string|max:255',
            'nombre_niveau' => 'nullable|integer',
            'file' => 'nullable|file|mimes:pdf|max:102400', // 100MB max
        ]);


        $operation = new TerrainOperation();
        $operation->id = Str::uuid();
        $operation->terrain_id = $request->terrain_id;
        $operation->operation_num = $request->operation_num;
        $operation->operation_motif = $request->operation_motif;
        $operation->date = $request->date_operation;
        $operation->operation_type = $request->operation_type;

        if ($request->operation_type == "TT_ET") {
            Terrain::where('id', $request->terrain_id)->update([
                'type' => 'Exonerations Temporaires'
            ]);

            $operation->operation_data = [
                'type_construction' => $request->type_construction,
                'nombre_niveau' => $request->nombre_niveau,
            ];
        }


        if ($request->operation_type == "ET_TT") {
            Terrain::where('id', $request->terrain_id)->update([
                'type' => 'Terrains taxables'
            ]);
        }




        if ($request->hasFile('file')) {

            $now = Carbon::now();
            $now->hour(23);
            $now->minute(59);
            $now->second(59);
            $terrain_document = new Terrain_document();

            if ($request->id != null) {
                $terrain_document = Terrain_document::where('id', $request->document_id)->get()->first();
            } else {
                $terrain_document->id = Str::uuid();
            }

            $file = $request->file('file');

            if ($file == null) {
            } else {
                $fileName = $file->getClientOriginalName();
                $file_extension = Str::upper($file->getClientOriginalExtension());
                $file_size = $file->getSize();
            }


            $terrain_document->filename = $fileName;
            $terrain_document->nom =  $fileName;
            $terrain_document->user_id =  Auth::user()->id;
            $terrain_document->date_document = $now->format('Y-m-d');
            $terrain_document->file_type = $file_extension;
            $terrain_document->file_size = $file_size;
            $terrain_document->decision = 'Validé';
            $terrain_document->terrain_id = $request->terrain_id;
            $terrain_document->operation_id  = $operation->id;
            $terrain_document->type_document = "EXONERATION_TEMPORAIRE";
            $terrain_document->save();

            if ($request->date_document != null) {
                $terrain_document->formated_date_document = $terrain_document->formated_date_document();
            }
            $terrain_document->formated_created_at = $terrain_document->formated_created_at();

            $file->move("files/terrains_nus_docs/" . $terrain_document->id . "/", $fileName);
        }

        $operation->save();

        return response()->json([
            'success' => 1,
            'operation_id' => $operation->id,
        ]);
    }

    public static function formatBytes($size, $precision = 2)
    {
        if ($size > 0) {
            $size = (int) $size;
            $base = log($size) / log(1024);
            $suffixes = array(' bytes', ' KB', ' MB', ' GB', ' TB');

            return round(pow(1024, $base - floor($base)), $precision) . $suffixes[floor($base)];
        } else {
            return $size;
        }
    }


    public function upload(Request $request)
    {
        $now = Carbon::now();
        $now->hour(23);
        $now->minute(59);
        $now->second(59);
        $terrain_document = new Terrain_document();

        if ($request->id != null) {
            $terrain_document = Terrain_document::where('id', $request->id)->get()->first();
        } else {
            $terrain_document->id = Str::uuid();
        }

        $file = $request->file('file');

        if ($file == null) {
        } else {
            $fileName = $file->getClientOriginalName();
            $file_extension = Str::upper($file->getClientOriginalExtension());
            $file_size = $file->getSize();
        }


        $terrain_document->filename = $fileName;
        $terrain_document->nom =  $fileName;
        $terrain_document->user_id =  Auth::user()->id;
        $terrain_document->date_document = $now->format('Y-m-d');
        $terrain_document->file_type = $file_extension;
        $terrain_document->file_size = $file_size;
        $terrain_document->decision = 'Validé';
        $terrain_document->type_document = $request->type_document;
        if ($request->filled('numero')) {
            $terrain_document->numero = $request->numero;
        }
        $terrain_document->save();

        if ($request->date_document != null) {
            $terrain_document->formated_date_document = $terrain_document->formated_date_document();
        }
        $terrain_document->formated_created_at = $terrain_document->formated_created_at();

        $file->move("files/terrains_nus_docs/" . $terrain_document->id . "/", $fileName);
        return Response::json(['success' => 1, 'document' => $terrain_document]);
    }


    public function file(string $file_id)
    {
        $terrain_document = Terrain_document::where('id', $file_id)->get()->first();
        $headers = array(
            'Content-Type: application/pdf',
        );

        return Response::download($terrain_document->documentPath(), $terrain_document->filename, $headers);
    }

    public function  destroy(Request $request)
    {
        $req = $request->all();

        $demande_id = $req['demande_id'];

        $demande = Demande::where('id', $demande_id)->get()->first();
        $dossier = Dossier::where('demande_id', $demande_id)->get()->first();

        if ($demande->user_id == Auth::user()->id) {
            $depot = Dossier_document::where('dossier_id', $dossier->id)->count();
            if ($depot > 0) {
                return Response::json(['success' => 0, 'error' => 'DOSSIER_WITH_DEPOT']);
            } else {

                foreach ($demande->petitionnaires as $petitionnaire) {
                    $petitionnaire->delete();
                }
                foreach ($demande->documents as $document) {
                    $document->delete();
                }
                $demande->delete();
                $dossier->delete();

                return Response::json(['success' => 1, 'error' => '']);
            }
        } else {
            return Response::json(['success' => 0, 'error' => 'NOT_AUTHORIZED']);
        }
    }


    private function calculateSanctions($year, $montant)
    {
        // Calculate months from March 1st of the year to current date
        $currentYear = now()->year;
        $marchFirst = Carbon::createFromDate($currentYear, 3, 1);
        $currentDate = now();
        $monthsFromMarch = 0;

        $monthsFromMarch = (int) $marchFirst->diffInMonths($currentDate);

        $yearsPassed = $currentYear - $year;
        $totalMonths = $monthsFromMarch + ($yearsPassed * 12);


        return [
            'details' => [
                ['name' => 'Défaut de déclaration de terrain', 'amount' => max(0.15 * $montant, 500)],
                ['name' => 'Paiement tardif', 'amount' => 0.10 * $montant],
                ['name' => 'Majoration pour le premier mois', 'amount' => 0.05 * $montant],
                ['name' => 'Majoration pour chaque mois supplémentaires', 'amount' => 0.005 * $montant * $totalMonths],
            ],
            'total_sanctions' => max(0.15 * $montant, 500) + 0.10 * $montant + 0.05 * $montant + (0.005 * $montant * $totalMonths),
            'months_passed' => $totalMonths,
        ];
    }

    private function calculateDelai($motif, $surface)
    {

        if (strtolower($motif) == 'construction') {
            return 3;
        }
        
        if (strtolower($motif) == 'lotissement') {
            $surfaceHa = $surface / 10000;
            
            if ($surfaceHa <= 20) {
                return 3;
            } elseif ($surfaceHa > 20 && $surfaceHa <= 100) {
                return 5;
            } elseif ($surfaceHa > 100 && $surfaceHa <= 250) {
                return 7;
            } elseif ($surfaceHa > 250 && $surfaceHa <= 400) {
                return 10;
            } else {
                return 15;
            }
        }
        
        return 0;
    }
}
