<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class DeclarationNumber extends Model
{
    use HasFactory;

    protected $fillable = [
        'year',
        'last_number'
    ];

    protected $casts = [
        'year' => 'integer',
        'last_number' => 'integer'
    ];

    /**
     * Get the next declaration number for the current year
     */
    public static function getNextNumber(?int $year = null): string
    {
        $year = $year ?? now()->year;
        
        return DB::transaction(function () use ($year) {
            // Get or create the declaration number record for the year
            $declarationNumber = self::firstOrCreate(
                ['year' => $year],
                ['last_number' => 0]
            );
            
            // Increment the number
            $declarationNumber->increment('last_number');
            
            // Format the number (D00001/2025)
            return sprintf('D%05d/%d', $declarationNumber->last_number, $year);
        });
    }

    /**
     * Get the current number for a specific year without incrementing
     */
    public static function getCurrentNumber(?int $year = null): string
    {
        $year = $year ?? now()->year;
        
        $declarationNumber = self::where('year', $year)->first();
        $currentNumber = $declarationNumber ? $declarationNumber->last_number : 0;
        
        return sprintf('D%05d/%d', $currentNumber, $year);
    }

    /**
     * Get the next number that would be generated (preview) without incrementing
     */
    public static function getNextNumberPreview(?int $year = null): string
    {
        $year = $year ?? now()->year;
        
        $declarationNumber = self::where('year', $year)->first();
        $nextNumber = $declarationNumber ? $declarationNumber->last_number + 1 : 1;
        
        return sprintf('D%05d/%d', $nextNumber, $year);
    }

}